package net.minecraft.src;

import java.io.File;
import java.text.DecimalFormat;
import java.lang.Math;

import java.util.Arrays;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import javax.imageio.ImageIO;

import net.minecraft.client.Minecraft;

import org.lwjgl.opengl.GL11;

// contains all the useful functions for writing to the display
// box drawing, text rendering, etc.
// along with helpers, such as HSV to RGB conversion

public class FruitDisplayLib
{
  
  // I keep making more and more variables and functions static
  // this might not be a good idea, but I do want it to compile...
  public static int[] charWidth = new int[256];
  public static int fontTextureName = 0;
  
  public static byte[] glyphWidth = new byte[65536];
  public final static int[] glyphTextureName = new int[256];
  public static int boundTextureName;
  
  private static float glyph_page_width = 256;
  // we are going to assume that the glyph pages are square
  // all the vanilla ones are
  
  private static RenderEngine renderEngine;
  // we'll make our font rendering code accessible to all
  // because screw you, private adjustable size font renderer
  
  private static boolean is_initialised = false;
  
  private static Minecraft minecraft;
  
  
  // changelog:
  
  // 1.1:
  // -added some resilience to Unicode characters - they now get displayed as question marks
  
  // 1.2a:
  // -altered the DrawShadedRect function to use direct GL calls instead of Tessellator.class calls
  // --this should fix the problem with only half of it being drawn
  // --along with providing a small speedup
  // -functions such as DrawShadedRect now take floats where they previously took doubles
  // --this was done because GL calls take floats and as such reduces the amount of typecasting I need to do
  
  public static String getVersion()
  {
    return "1.2a [MC r1.2.5]";
    // one anti-unicode hack, coming right up
  }
  
  
  public static void initialiseGlyphWidths()
  {
    BufferedImage font_texture;
    try
    {
      // to do: make it use the fonts from a texture pack...
      font_texture = ImageIO.read(RenderEngine.class.getResourceAsStream("/font/default.png"));
      InputStream glyph_width_stream = RenderEngine.class.getResourceAsStream("/font/glyph_sizes.bin");
      glyph_width_stream.read(glyphWidth);
    }
    catch (IOException glyph_size_fault)
    {
      throw new RuntimeException(glyph_size_fault);
    }
    
    
    // we just copy this straight from the FontRenderer sources
    int char_col;
    int char_row;
    int char_width;
    int font_tex_xpos;
    int var16;
    
    int font_page_width = font_texture.getWidth();
    int font_page_height = font_texture.getWidth();
    int[] pixelmap = new int[font_page_width * font_page_height];
    
    font_texture.getRGB(0, 0, font_page_width, font_page_height, pixelmap, 0, font_page_width);
    
    for(int i = 0; i < 256; i++)
    {
      char_col = i % 16;
      char_row = i / 16;
      char_width = 7;
      
      while (true)
      {
	if (char_width >= 0)
	{
	  font_tex_xpos = char_col * 8 + char_width;
	  boolean var14 = true;
	  
	  for (int j = 0; j < 8 && var14; j++)
	  {
	    var16 = (char_row * 8 + j) * font_page_width;
	    int var17 = pixelmap[font_tex_xpos + var16] & 255;
	    
	    if (var17 > 0)
	    {
	      var14 = false;
	    }
	  }
	  
	  if (var14)
	  {
	    char_width--;
	    continue;
	  }
	}
	
	if (i == 32)
	{
	  char_width = 2;
	}
	
	charWidth[i] = char_width + 2;
	break;
      }
    }
    
    fontTextureName = renderEngine.allocateAndSetupTexture(font_texture);
    
  }  
  
  
  // par1 = glyph page index
  // var3 = glyph page filename
  // var2 = glyph page texture
  // var5 - glyph page exception
  
  private static void loadGlyphTexture(int glyph_page_index)
  {
    String glyph_page_filename = String.format("/font/glyph_%02X.png", new Object[] {Integer.valueOf(glyph_page_index)});
    BufferedImage glyph_page_texture;
    
    /*
    try
    {
      glyph_page_texture = ModLoader.loadImage(renderEngine, glyph_page_filename);
      glyph_page_width = (float)glyph_page_texture.getWidth();
    }
    catch(Exception glyph_page_error)
    {
      throw new RuntimeException(glyph_page_error);
    }
    */
    
    // the line above was supposed to give us texture pack compatibility for fonts
    // but it fails to initialise the font graphics properly
    // we'll load the glyphs in the same way as the default FontRenderer, then
    
    
      try
      {
        glyph_page_texture = ImageIO.read(RenderEngine.class.getResourceAsStream(glyph_page_filename));
	// glyph_page_width = (float)glyph_page_texture.getWidth();
     }
     catch (IOException var5)
     {
       throw new RuntimeException(var5);
     }
    
    glyphTextureName[glyph_page_index] = renderEngine.allocateAndSetupTexture(glyph_page_texture);
    boundTextureName = glyphTextureName[glyph_page_index];
  }
  
  
  // perhaps make this function private
  // and have a public "renderScaledText" function which accepts: string, xpos, ypos, scale_factor
  // perhaps pass in R, G and B colour parameters too?
  // attempt by putting a GL11.glColor4f declaration before the render code?
  
  private static float renderScaledUnicodeChar(char unicode_char, float posX, float posY, float font_scaling, float[] RGBarray, float alpha)
  {
    if (glyphWidth[unicode_char] == 0)
    {
      return 0.0F;
    }
    else
    {
      
      // if we want to change the colour of rendered text, this is where we do it
      // by creating a GL11.glColor4f(R,G,B,A) here
      
      GL11.glColor4f(RGBarray[0], RGBarray[1], RGBarray[2], alpha);
      
      int glyph_page = unicode_char / 256;
      
      if (glyphTextureName[glyph_page] == 0)
      {
	loadGlyphTexture(glyph_page);
      }
      
      if (boundTextureName != glyphTextureName[glyph_page])
      {
	GL11.glBindTexture(GL11.GL_TEXTURE_2D, glyphTextureName[glyph_page]);
	boundTextureName = glyphTextureName[glyph_page];
      }
      
      // 4 times bitshifting to the right, ignoring sign
      // read: delete the lower 4 characters, shift the next 4 in their place
      // so, var4 = upper nibble
      int upper_width_nibble = glyphWidth[unicode_char] >>> 4;
      // logical AND with 15, or basically erasing all but the lower 4 bits
      // so, var5 = lower nibble
      int lower_width_nibble = glyphWidth[unicode_char] & 15;
      
      // below code filched from FontRenderer's getCharWidth function
      // thought it would allow me to remove the fudge factors, but no...
      /*
      if(lower_width_nibble > 7)
      {
	lower_width_nibble = 15;
	upper_width_nibble = 0;
      }
      */
      
      float var6 = (float)upper_width_nibble;
      float var7 = (float)(lower_width_nibble + 1.0F);
      // these bits above are going to cause real problems if I ever fully adapt it
      // to deal with higher-resolution fonts
      float var8 = (float)(unicode_char % 16 * (glyph_page_width / 16.0F)) + var6; // - 3.0F;
      // var8 = fractional co-ordinate of glyph on the x-axis
      // that - 2.0F on the end is a bastard empirical hack, since the initial code missed out
      // the first few pixels of the font
      float var9 = (float)((unicode_char & 255) / 16 * (glyph_page_width / 16.0F));
      // var9 = fractional co-ordinate of glyph on the y-axis
      float var10 = var7 - var6 - 0.02F;
      // float var10 = var7 - var6 + 4.0F;
      // likewise, this + 4.0F on the end is because it missed out the pixels at the end
      // var10 appears to be the width of the character
      GL11.glBegin(GL11.GL_TRIANGLE_STRIP);
      // scaling for altered resolution texture packs
      // each glyph page is 256x256 pixels across and contains a 16x16 array of glyphs
      // so we need to fetch the size of said glyph page
      // easily done with the int getWidth() function in BufferedImage
      // we store it as a float, for ease of multiplication
      GL11.glTexCoord2f(var8 / glyph_page_width, var9 / glyph_page_width);
      GL11.glVertex3f(posX , posY, 0.0F);
      GL11.glTexCoord2f(var8 / glyph_page_width, (var9 + 15.98F) / glyph_page_width);
      GL11.glVertex3f(posX, posY + (7.99F * font_scaling), 0.0F);
      GL11.glTexCoord2f((var8 + var10) / glyph_page_width, var9 / glyph_page_width);
      GL11.glVertex3f(posX + (var10 / 2.0F * font_scaling), posY, 0.0F);
      GL11.glTexCoord2f((var8 + var10) / glyph_page_width, (var9 + 15.98F) / glyph_page_width);
      GL11.glVertex3f(posX + (var10 / 2.0F * font_scaling), posY + (7.99F * font_scaling), 0.0F);
      // though there are some hard-coded variables above, we should be alright
      // note that it's not completely image size independent, though
      GL11.glEnd();
      return ((var7 - var6) / 2.0F + 1.0F) * font_scaling;
      // return ((var7 - var6) / 1.3F + 1.0F) * font_scaling;
      
      // return value is what we add on to the text position to space letters properly
    }
  }
  
  private static float renderScaledChar(int default_char, float posX, float posY, float font_scaling, float[] RGBarray, float alpha, boolean hasShadow)
  {
    
    // unicode hack 1
    if(default_char > 255)
    {
      default_char = 63;
      // 63 = 0x3f = question mark
      // return 0;
    }
    
    GL11.glColor4f(RGBarray[0], RGBarray[1], RGBarray[2], alpha);
    
    float var3 = (float)(default_char % 16 * 8);
    float var4 = (float)(default_char / 16 * 8);
    
    //if (boundTextureName != minecraft.fontRenderer.fontTextureName)
    //{
      // GL11.glBindTexture(GL11.GL_TEXTURE_2D, minecraft.fontRenderer.fontTextureName);
      GL11.glBindTexture(GL11.GL_TEXTURE_2D, fontTextureName);
      //boundTextureName = minecraft.fontRenderer.fontTextureName;
    //}
    
      
    float shadowExtra = hasShadow ? 1.0F : 0.0F;
    float var6 = (float)charWidth[default_char] - 0.01F;
    GL11.glBegin(GL11.GL_TRIANGLE_STRIP);
    GL11.glTexCoord2f(var3 / 128.0F, var4 / 128.0F);
    GL11.glVertex3f(posX + shadowExtra * font_scaling, posY, 0.0F);
    GL11.glTexCoord2f(var3 / 128.0F, (var4 + 7.99F) / 128.0F);
    GL11.glVertex3f(posX - shadowExtra * font_scaling, posY + (7.99F * font_scaling), 0.0F);
    GL11.glTexCoord2f((var3 + var6) / 128.0F, var4 / 128.0F);
    GL11.glVertex3f(posX + (var6 + shadowExtra) * font_scaling, posY, 0.0F);
    GL11.glTexCoord2f((var3 + var6) / 128.0F, (var4 + 7.99F) / 128.0F);
    GL11.glVertex3f(posX + (var6 - shadowExtra) * font_scaling, posY + (7.99F * font_scaling), 0.0F);
    GL11.glEnd();
    return (float)charWidth[default_char] * font_scaling;
  }
  
  
  public static void renderScaledStringAtPos(String input, float xstart, float ystart, float font_scaling, float[] RGBarray, float alpha, boolean hasShadow)
  {
    
    // shadow is broken for text sizes smaller than the default
    // you should draw a light grey rectangle behind it instead
    
    float writing_pos = xstart;
    float writing_offset = 0.0F;
    for(int i = 0; i < input.length(); i++)
    {
      if(hasShadow)
      {
	renderScaledChar(input.charAt(i), writing_pos, ystart, font_scaling, HSVtoRGB(0.0F, 0.0F, 0.0F), alpha, true);
	// renderScaledUnicodeChar(input.charAt(i), writing_pos, ystart, font_scaling, HSVtoRGB(0.0F, 0.0F, 0.0F), alpha);
	// yes, I should probably define a constant array of {0.0F, 0.0F, 0.0F} and pass that
	// it would speed execution minutely, but meh, lazy...
      }
      writing_offset = renderScaledChar(input.charAt(i), writing_pos, ystart, font_scaling, RGBarray, alpha, false);
      // writing_offset = renderScaledUnicodeChar(input.charAt(i), writing_pos, ystart, font_scaling, RGBarray, alpha);
      writing_pos += writing_offset;
      // was originally renderScaledUnicodeChar
      // but it completely refuses to get the correct texture co-ordinates
      // so we'll just use the default font instead...
    }
  }
  
  
  // originally from Gui.java
  // right, now it's getting the colour and transparency right
  // but not drawing the full rectangle
  // like this, it draws the upper-left, upper-right and lower-left vertices
  // so it missed out the last one
  public static void drawShadedRect(float xpos, float ypos, float xsize, float ysize, float[] RGBarray, float alpha)
  {
    // it should be noted that this is for HUD elements only
    // they are given a Z-depth of zero
    
    Tessellator tess = Tessellator.instance;
    GL11.glEnable(GL11.GL_BLEND);
    GL11.glDisable(GL11.GL_TEXTURE_2D);
    GL11.glBlendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);
    GL11.glColor4f(RGBarray[0], RGBarray[1], RGBarray[2], alpha);
    
    GL11.glBegin(GL11.GL_QUADS);
    GL11.glVertex3f(xpos, ypos, 0.0F);
    GL11.glVertex3f(xpos, ypos + ysize, 0.0F);
    GL11.glVertex3f(xpos + xsize, ypos + ysize, 0.0F);
    GL11.glVertex3f(xpos + xsize, ypos, 0.0F);
    GL11.glEnd();
    
    
    /*
    tess.startDrawingQuads();
    tess.addVertex(xpos, ypos, 0.0D);
    tess.addVertex(xpos, ypos + ysize, 0.0D);
    tess.addVertex(xpos + xsize, ypos, 0.0D);
    tess.addVertex(xpos + xsize, ypos + ysize, 0.0D);
    tess.draw();
    */
    
    // time for a hideously disgusting hack
    // since Tesselator won't draw a quad even when explicitly commanded to
    // (it gives a triangle, despite the function name)
    // we'll need to do it as 2 triangles...
    // the above does the top-left, top-right and bottom-left vertices
    // naturally, it won't do anything when fed 3 vertices
    // fuck you, Tesselator
    
    // so we need top-right, bottom-left and bottom-right
    // to get this, cyclic permute the whole list upwards (to put the first one on the bottom)
    // then swap the top two elements from the new list
    
    // Notch, your code sucks donkey balls
    /*
    tess.startDrawingQuads();
    tess.addVertex(xpos + xsize, ypos, 0.0D);
    tess.addVertex(xpos, ypos + ysize, 0.0D);
    tess.addVertex(xpos + xsize, ypos + ysize, 0.0D);
    tess.addVertex(xpos, ypos, 0.0D);
    tess.draw();
    */
    
    GL11.glEnable(GL11.GL_TEXTURE_2D);
    GL11.glDisable(GL11.GL_BLEND);
  }
  
  // we now need one to draw a hollow rectangle
  // simplest is to call DrawShadedRect 4 times
  
  public static void drawHollowRect(float xpos, float ypos, float xsize, float ysize, float thickness, float[] RGBarray, float alpha)
  {
    // draw each "line" in turn, making sure not to overlap
    // otherwise the corners will end up with half the transparency and look weird
    // top line
    drawShadedRect(xpos, ypos, xsize - thickness, thickness, RGBarray, alpha);
    
    // right line
    drawShadedRect(xpos + xsize - thickness, ypos, thickness, ysize - thickness, RGBarray, alpha);
    
    // bottom line
    drawShadedRect(xpos + thickness, ypos + ysize - thickness, xsize - thickness, thickness, RGBarray, alpha);
    
    // left line
    drawShadedRect(xpos, ypos + thickness, thickness, ysize - thickness, RGBarray, alpha);
  }
  
  
  
  // we'll be giving the item condition as a hue
  // this function will make my life a lot easier
  
  public static float[] HSVtoRGB(float hue, float saturation, float value)
  {
    // acceptable input values: hue = 0 to 360, saturation, value = 0 to 1
    
    // from the wiki page on HSV colour space:
    // [chroma] C = V * S
    // [hue_prime] H' = H / 60 (in degrees)
    // [alt_component] X = C(1 - abs((H' % 2) - 1))
    // [value_match] m = V - C
    // (R_1, G_1, B_1) = (C, X, 0) for 0 =< H' < 1
    // (X, C, 0) for 1 =< H' < 2
    // (0, C, X) for 2 =< H' < 3
    // (0, X, C) for 3 =< H' < 4
    // (X, 0, C) for 4 =< H' < 5
    // (C, 0, X) for 5 =< H' < 6
    // (0, 0, 0) otherwise, since it's undefined
    // (R, G, B) = (R_1 + m, G_1 + m, B_1 + m)
    
    // in fact, let's assign "otherwise" to (1, 1, 1)
    // so that we can take advantage of the "undefined" or NULL output from utility armours
    // RGB values outputted thus will range from 0 to 1 - perfect for my needs
    
    float[] result = {1.0F, 1.0F, 1.0F};
    // we leave the default as white to avoid putting in another if check
    // not technically correct, but it makes my life a little bit easier
    // this means that we can set <item>_fraction to 1.1F
    // and get white text - good for non-damageable items
    
    
    if(hue == 0.0F)
    {
      result[0] = 0.0F;
      result[1] = 0.0F;
      result[2] = 0.0F;
      return result;
    }
    // another break from technical correctness here
    // when hue is exactly zero (in the case of depleted armour)
    // we'll return black instead of the FF0000 we would normally have gotten
    
    float chroma = value * saturation;
    float hue_prime = hue / 60.0F;
    float alt_component = chroma * (1.0F - Math.abs((hue_prime % 2.0F) - 1.0F));
    
    float value_match = value - chroma;
    
    boolean isValidColour = true;
    
    if(0.0F <= hue_prime && hue_prime < 1.0F)
    {
      result[0] = chroma;
      result[1] = alt_component;
      result[2] = 0.0F;
    }
    else if(1.0F <= hue_prime && hue_prime < 2.0F)
    {
      result[0] = alt_component;
      result[1] = chroma;
      result[2] = 0.0F;
    }
    else if(2.0F <= hue_prime && hue_prime < 3.0F)
    {
      result[0] = 0.0F;
      result[1] = chroma;
      result[2] = alt_component;
    }
    else if(3.0F <= hue_prime && hue_prime < 4.0F)
    {
      result[0] = 0.0F;
      result[1] = alt_component;
      result[2] = chroma;
    }
    else if(4.0F <= hue_prime && hue_prime < 5.0F)
    {
      result[0] = alt_component;
      result[1] = 0.0F;
      result[2] = chroma;
    }
    else if(5.0F <= hue_prime && hue_prime < 6.0F)
    {
      result[0] = chroma;
      result[1] = 0.0F;
      result[2] = alt_component;
    }
    else
    {
      isValidColour = false;
    }
    
    if(isValidColour)
    {
      for(int i = 0; i < 3; i++)
      {
	result[i] = result[i] + value_match;
	// I really miss C++'s STL vector class
	// because I could use i < result.size() here for the ending condition
      }
    }
    
    return result;
  }
  
  
  // I think that OptiFine is overwriting getStringWidth and/or getCharWidth from FontRenderer
  // since Minecraft thinks that the actual string length changes if you use a texture pack
  // so I'll just copy out the unmodified ones and stick them here
  
  public static int getStringWidth(String par1Str)
  {
    if (par1Str == null)
    {
      return 0;
    }
    else
    {
      int var2 = 0;
      boolean var3 = false;
      
      for (int var4 = 0; var4 < par1Str.length(); ++var4)
      {
	char var5 = par1Str.charAt(var4);
	int var6 = getCharWidth(var5);
	
	if (var6 < 0 && var4 < par1Str.length() - 1)
	{
	  ++var4;
	  var5 = par1Str.charAt(var4);
	  
	  if (var5 != 108 && var5 != 76)
	  {
	    if (var5 == 114 || var5 == 82)
	    {
	      var3 = false;
	    }
	  }
	  else
	  {
	    var3 = true;
	  }
	  
	  var6 = getCharWidth(var5);
	}
	
	var2 += var6;
	
	if (var3)
	{
	  ++var2;
	}
      }
      
      return var2;
    }
  }
  
  /**
   * Returns the width of this character as rendered.
   */
  public static int getCharWidth(char par1)
  {
    
    if(par1 > 255)
    {
      par1 = 63;
      // a question mark, 0x3F
      // new unicode hack - let's have a display this time
    }
    
    if (par1 == 167)
    {
      return -1;
    }
    /*
    else if(par1 > 255)
    {
      // unicode hack 2
      return 0;
    }
    */
    else
    {
      int var2 = ChatAllowedCharacters.allowedCharacters.indexOf(par1);
      
      // need to remove the unicode flag
      // shame, really - I'd like it to be multi-language compatible
      
      if (var2 >= 0)
      {
	return charWidth[var2 + 32];
      }
      else
      {
	return 0;
      }
    }
  }
  
  
  
  public static void init()
  {
    if(!is_initialised)
    {
      renderEngine = ModLoader.getMinecraftInstance().renderEngine;
      
      initialiseGlyphWidths();
      
      minecraft = ModLoader.getMinecraftInstance();
      
      System.out.println("Fruit Display Library v" + getVersion() + " initialised.");
      
      is_initialised = true;
    }
  }
  
}
