package lostcoder.ic2.nuclearpan;

import ic2.api.IReactor;
import ic2.api.IReactorChamber;

import java.util.Random;

import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemFood;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.FurnaceRecipes;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraftforge.common.ForgeDirection;
import net.minecraftforge.common.ISidedInventory;

public class TileEntityNuclearPan extends TileEntity implements ISidedInventory
{
    private Random rand = new Random();
    public ItemStack[] inventory;
    public IReactor reactor;
    public int heat;
    public int progress;

    public TileEntityNuclearPan()
    {
        super();
        inventory = new ItemStack[2];
        reactor = null;
        heat = 0;
        progress = 0;
    }

    /**
     * Checks whether item stack is food.
     */
    public static boolean isFood(ItemStack stack)
    {
        return (stack.getItem() instanceof ItemFood) || NuclearPan.inWhiteList(stack);
    }

    /**
     * Returns true if the pan can cook an item, i.e. has a source item, destination stack isn't full, etc.
     */
    public boolean canCook()
    {
        if (inventory[0] == null) return false;
        if (!isFood(inventory[0])) return false;

        ItemStack stack = FurnaceRecipes.smelting().getSmeltingResult(inventory[0]);
        if (stack == null) return false;

        // Ash piles don't need this part ;)
        if (heat <= NuclearPan.instance.maxHeat)
        {
            // Check whether output slot can fit smelting result
            if (inventory[1] == null) return true;
            if (!inventory[1].isItemEqual(stack)) return false;
            int result = inventory[1].stackSize + stack.stackSize;
            return (result <= getInventoryStackLimit() && result <= stack.getMaxStackSize());
        }
        return true;
    }

    /**
     * Turn one item from the pan source stack into the appropriate cooked item in the pan result stack
     */
    public void cookItem()
    {
        // Check heat level
        if (heat >= NuclearPan.instance.maxHeat)
        {
            // Cook ash pile if overheat
            ItemStack stack = new ItemStack(NuclearPan.instance.ashPile);
            if (inventory[1] == null)
            {
                inventory[1] = stack.copy();
            }
            else if (inventory[1].isItemEqual(stack) && inventory[1].stackSize < getInventoryStackLimit())
            {
                inventory[1].stackSize++;
            }
            else
            {
                // Drop ash pile into world
                EntityItem entity = new EntityItem(worldObj, xCoord + 0.5F, yCoord + 0.25F, zCoord + 0.5F, stack.copy());
                if (stack.hasTagCompound())
                {
                    entity.getEntityItem().setTagCompound((NBTTagCompound)stack.getTagCompound().copy());
                }
                entity.motionX = (float)rand.nextGaussian() * 0.05F;
                entity.motionY = (float)rand.nextGaussian() * 0.25F;
                entity.motionZ = (float)rand.nextGaussian() * 0.05F;
                worldObj.spawnEntityInWorld(entity);
            }
        }
        else
        {
            if (!canCook()) return;

            ItemStack stack = FurnaceRecipes.smelting().getSmeltingResult(inventory[0]);

            if (inventory[1] == null)
            {
                inventory[1] = stack.copy();
            }
            else if (inventory[1].isItemEqual(stack) && (inventory[1].stackSize + stack.stackSize <= stack.getMaxStackSize()))
            {
                inventory[1].stackSize += stack.stackSize;
            }
        }
        inventory[0].stackSize--;
        if (inventory[0].stackSize <= 0) inventory[0] = null;
    }

    @Override
    public void updateEntity()
    {
        super.updateEntity();

        // Sentinel
        if (isInvalid()) return;

        // Done for client
        if (worldObj.isRemote) return;

        // Try to find reactor
        if (reactor == null && yCoord > 0)
        {
            TileEntity tile = worldObj.getBlockTileEntity(xCoord, yCoord-1, zCoord);
            if (tile != null)
            {
                if (tile instanceof IReactor)
                {
                    reactor = (IReactor) tile;
                }
                else if (tile instanceof IReactorChamber)
                {
                    reactor = ((IReactorChamber) tile).getReactor();
                }
            }
        }

        // Obtain heat if reactor present
        if (reactor != null)
        {
            try
            {
                heat = reactor.getHeat();
            }
            catch (Exception e)
            {
                heat = 0;
                reactor = null;
            }
        }
        else
        {
            heat = 0;
        }

        // Check whether we can cook input item
        if (canCook())
        {
            // Collect heat
            if (heat >= NuclearPan.instance.minHeat)
            {
                progress += heat - NuclearPan.instance.minHeat + 1;
            }

            // Check whether there is enough heat to cook items
            if (progress >= NuclearPan.instance.heatPerItem)
            {
                do
                {
                    cookItem();
                    progress -= NuclearPan.instance.heatPerItem;
                }
                while (canCook() && progress >= NuclearPan.instance.heatPerItem);
            }
        }

        // Reset progress if cannot cook
        if (!canCook())
        {
            progress = 0;
        }
    }

    /**
     * Reads a tile entity from NBT.
     */
    @Override
    public void readFromNBT(NBTTagCompound nbt)
    {
        super.readFromNBT(nbt);

        // Read inventory stacks from NBT.

        NBTTagList items = nbt.getTagList("Items");
        inventory = new ItemStack[getSizeInventory()];

        for (int i = 0; i < items.tagCount(); i++)
        {
            NBTTagCompound item = (NBTTagCompound) items.tagAt(i);
            byte slot = item.getByte("Slot");

            if (slot >= 0 && slot < inventory.length)
            {
                inventory[slot] = ItemStack.loadItemStackFromNBT(item);
            }
        }

        // Read rest parameters

        progress = nbt.getInteger("progress");
    }

    /**
     * Writes a tile entity to NBT.
     */
    @Override
    public void writeToNBT(NBTTagCompound nbt)
    {
        super.writeToNBT(nbt);

        // Write inventory stacks to NBT.

        NBTTagList items = new NBTTagList();

        for (int i = 0; i < inventory.length; ++i)
        {
            if (inventory[i] != null)
            {
                NBTTagCompound item = new NBTTagCompound();
                item.setByte("Slot", (byte) i);
                inventory[i].writeToNBT(item);
                items.appendTag(item);
            }
        }

        nbt.setTag("Items", items);

        // Write rest parameters

        nbt.setInteger("progress", progress);
    }

    // IInventory implementation

    /**
     * Returns the number of slots in the inventory.
     */
    @Override
    public int getSizeInventory()
    {
        return inventory.length;
    }

    /**
     * Returns the stack in slot i
     */
    @Override
    public ItemStack getStackInSlot(int i)
    {
        return inventory[i];
    }

    /**
     * Removes from an inventory slot (first arg) up to a specified number (second arg) of items and returns them in a
     * new stack.
     */
    @Override
    public ItemStack decrStackSize(int i, int count)
    {
        if (inventory[i] == null) return null;

        ItemStack stack;

        if (inventory[i].stackSize <= count)
        {
            stack = inventory[i];
            inventory[i] = null;
            return stack;
        }
        else
        {
            stack = inventory[i].splitStack(count);
            if (inventory[i].stackSize == 0) inventory[i] = null;
            return stack;
        }
    }

    /**
     * When some containers are closed they call this on each slot, then drop whatever it returns as an EntityItem -
     * like when you close a workbench GUI.
     */
    @Override
    public ItemStack getStackInSlotOnClosing(int i)
    {
        ItemStack stack = inventory[i];
        inventory[i] = null;
        return stack;
    }

    /**
     * Sets the given item stack to the specified slot in the inventory (can be crafting or armor sections).
     */
    @Override
    public void setInventorySlotContents(int i, ItemStack stack)
    {
        inventory[i] = stack;

        if (stack != null && stack.stackSize > getInventoryStackLimit())
        {
            stack.stackSize = getInventoryStackLimit();
        }
    }

    /**
     * Returns the name of the inventory.
     */
    @Override
    public String getInvName()
    {
        return "Nuclear Pan";
    }

    /**
     * Returns the maximum stack size for a inventory slot. Seems to always be 64, possibly will be extended. *Isn't
     * this more of a set than a get?*
     */
    @Override
    public int getInventoryStackLimit()
    {
        return 64;
    }

    /**
     * Do not make give this method the name canInteractWith because it clashes with Container
     */
    @Override
    public boolean isUseableByPlayer(EntityPlayer player)
    {
        return worldObj.getBlockTileEntity(xCoord, yCoord, zCoord) != this ? false : player.getDistanceSq(xCoord + 0.5D, yCoord + 0.5D, zCoord + 0.5D) <= 64.0D;
    }

    @Override
    public void openChest() {}

    @Override
    public void closeChest() {}

    @Override
    public int getStartInventorySide(ForgeDirection side)
    {
        if (side == ForgeDirection.UP) return 0;
        return 1;
    }

    @Override
    public int getSizeInventorySide(ForgeDirection side)
    {
        return 1;
    }
}
