package mods.NetworkAnchor;

import ic2.api.Direction;
import ic2.api.energy.event.EnergyTileLoadEvent;
import ic2.api.energy.event.EnergyTileUnloadEvent;
import ic2.api.energy.tile.IEnergySink;
import ic2.api.energy.tile.IEnergyTile;
import ic2.api.item.ElectricItem;
import ic2.api.item.IElectricItem;
import ic2.api.network.INetworkClientTileEntityEventListener;
import ic2.api.network.INetworkDataProvider;
import ic2.api.network.INetworkUpdateListener;
import ic2.api.network.NetworkHelper;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.TreeSet;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.world.ChunkCoordIntPair;
import net.minecraftforge.common.ForgeChunkManager;
import net.minecraftforge.common.ForgeChunkManager.Ticket;
import net.minecraftforge.common.ForgeChunkManager.Type;
import net.minecraftforge.common.MinecraftForge;

public class TileEntityNetworkAnchor extends TileEntity implements IInventory, IEnergySink, INetworkClientTileEntityEventListener, INetworkDataProvider, INetworkUpdateListener
{
    private static final List<String> networkedFields = Arrays.asList(new String[] {"energyEnabled", "scan", "area", "tilesFound", "chunksForced", "chunksFound", "ticketsUsed", "energy", "maxEnergy"});
    public ItemStack[] inventory; // The ItemStacks that hold the items in upgrade slots
    public boolean initialized;
    public boolean addedToEnergyNet;
    public boolean energyEnabled;
    public boolean scan;
    public int area; // Radius of chunk grid in area mode. NxN, where N = 2 * area + 1.
    public int tilesFound;
    public int chunksForced;
    public int chunksFound;
    public int ticketsUsed;
    public int blocksPerTick;
    public int energy;
    public int maxEnergy;
    public int maxInput;
    public int tier;
    public List<Ticket> tickets;
    public TreeSet<BlockCoords> network, pending;
    public TreeSet<ChunkCoords> chunks;

    /**
     * States:<br>
     * 0 - initialization (decide which mode to use, generate chunk grid in area mode);<br>
     * 1 - network scanning, energy consumption;<br>
     * 2 or higher - work done, do nothing, consume energy;<br>
     */
    public int state;

    public TileEntityNetworkAnchor()
    {
        super();
        inventory = new ItemStack[4];
        initialized = false;
        addedToEnergyNet = false;
        energyEnabled = NetworkAnchor.energyEnabled;
        scan = false;
        area = 0; // default 1x1
        tilesFound = 0;
        chunksForced = 0;
        chunksFound = 0;
        ticketsUsed = 0;
        blocksPerTick = 1;
        energy = 0;
        maxEnergy = 10000;
        maxInput = 32;
        tier = 1;
        tickets = new ArrayList<Ticket>();
        network = new TreeSet<BlockCoords>();
        pending = new TreeSet<BlockCoords>();
        chunks = new TreeSet<ChunkCoords>();
        state = 0;
    }

    /**
     * Safe version of NetworkHelper.updateTileEntityField. Will work only after device requested initial data (initialized).
     *
     * @param field to synchronize.
     */
    public void syncField(String field)
    {
        if (initialized) NetworkHelper.updateTileEntityField(this, field);
    }

    /**
     * Networked setter for energy. Will trigger network sync if needed.
     *
     * @param value - new value for energy.
     */
    public void setEnergy(int value)
    {
        energy = value;
        if (energyEnabled) syncField("energy");
    }

    /**
     * Networked setter for maxEnergy. Will trigger network sync if needed.
     *
     * @param value - new value for maxEnergy.
     */
    public void setMaxEnergy(int value)
    {
        maxEnergy = value;
        if (energyEnabled) syncField("maxEnergy");
    }

    public void removeFromEnergyNet()
    {
        if (addedToEnergyNet == true)
        {
            MinecraftForge.EVENT_BUS.post(new EnergyTileUnloadEvent(this));
            addedToEnergyNet = false;
        }
    }

    @Override
    public void invalidate()
    {
        super.invalidate();
        removeFromEnergyNet();
        if (worldObj.isRemote == false)
        {
            cleanupGeometry();
            releaseChunks();
        }
    }

    @Override
    public void onChunkUnload()
    {
        removeFromEnergyNet();
        super.onChunkUnload();
    }

    public void init(boolean now)
    {
        // iterate through area and generate chunk grid
        for (int x = -area; x <= area; x++)
        {
            for (int z = -area; z <= area; z++)
            {
                chunks.add(new ChunkCoords(x * 16 + xCoord, z * 16 + zCoord));
            }
        }

        // network mode part
        if (scan == true)
        {
            // add self as starting point
            pending.add(new BlockCoords(xCoord, yCoord, zCoord));

            if (now == false)
            {
                state = 1; // schedule network scan
                return;
            }

            // immediately scan network
            while (!pending.isEmpty()) processPendingBlock();
        }
        forceChunks();
    }

    @Override
    public void updateEntity()
    {
        super.updateEntity();

        // Request initial data for client
        if (initialized == false)
        {
            NetworkHelper.requestInitialData(this);
            initialized = true;
        }

        // Make sure that we connected to energy net
        if (addedToEnergyNet == false)
        {
            MinecraftForge.EVENT_BUS.post(new EnergyTileLoadEvent(this));
            addedToEnergyNet = true;
        }

        // Done for client
        if (worldObj.isRemote) return;

        // Obtain energy from items, if energy consumption is enabled
        if (energyEnabled)
        {
            for (ItemStack stack : inventory)
            {
                if (energy >= maxEnergy) break; // stop if full
                IElectricItem electricItem = getBattery(stack);
                if (electricItem == null) continue;
                int amount = Math.min(electricItem.getTransferLimit(stack), maxEnergy - energy);
                amount = ElectricItem.manager.discharge(stack, amount, tier, false, false);
                setEnergy(energy + amount);
                if (amount != 0) break; // no more than one item at once
            }
        }

        // Work

        // State 0: wait for internal storage to be full, if energy consumption is enabled
        if (state == 0 && (energy >= maxEnergy || !energyEnabled)) init(false);

        // State 1: scan network
        if (state == 1)
        {
            for (int i = 0; i < blocksPerTick; i++) // process some pending blocks
            {
                if (pending.isEmpty())
                {
                    // work done - try to force chunks and finish.
                    forceChunks();
                    break;
                }
                processPendingBlock(); // process single pending block
            }
        }

        // State 2: work done - stay idle and consume energy
        if (state == 2 && energyEnabled)
        {
            // Calculate energy requirement
            int a = NetworkAnchor.energyBase;
            int b = NetworkAnchor.energyPerTile * tilesFound;
            int c = NetworkAnchor.energyPerChunk * chunksForced;
            int cost = a + b + c;
            int energy2 = energy - cost;

            if (energy2 >= 0)
                setEnergy(energy2);
            else
                reset(); // not enough energy - start over
        }
    }

    public Ticket newTicket()
    {
        Ticket ticket;

        // Quick fix for unistall issue
        // if (placedBy == null)
            ticket = ForgeChunkManager.requestTicket(NetworkAnchor.instance, worldObj, Type.NORMAL);
        // else
        //     ticket = ForgeChunkManager.requestPlayerTicket(NetworkAnchor.instance, placedBy.username, worldObj, Type.NORMAL);

        if (ticket != null)
        {
            NBTTagCompound nbt = ticket.getModData();
            nbt.setInteger("x", xCoord);
            nbt.setInteger("y", yCoord);
            nbt.setInteger("z", zCoord);
            tickets.add(ticket);
        }

        return ticket;
    }

    public void forceChunks()
    {
        Ticket ticket;
        int ticketIndex = 0;

        // collect statistics
        chunksFound = chunks.size();
        syncField("chunksFound");

        while (chunks.isEmpty() == false)
        {
            // pick existing ticket or make a new one
            if (ticketIndex < tickets.size())
                ticket = tickets.get(ticketIndex);
            else
                ticket = newTicket();

            // error check if no more tickets available
            if (ticket == null) break;

            for (int i = ticket.getMaxChunkListDepth(); i > 0; i--)
            {
                if (chunks.isEmpty()) break;

                // get chunk
                ChunkCoords cc = chunks.pollFirst();

                // force it
                ForgeChunkManager.forceChunk(ticket, new ChunkCoordIntPair(cc.x, cc.z));

                // collect statistics
                chunksForced++;
                syncField("chunksForced");
            }

            // next ticket
            ticketIndex++;
        }

        // free not used tickets if they present
        for (int i = tickets.size() - 1; i >= 0; i--)
        {
            ticket = tickets.get(i);
            if (ticket.getChunkList().isEmpty() == true)
            {
                tickets.remove(i);
                ForgeChunkManager.releaseTicket(ticket);
            }
        }

        // collect statistics
        ticketsUsed = tickets.size();
        syncField("ticketsUsed");

        // free memory from geometry - we don't need it any more
        cleanupGeometry();

        // work done
        state = 2;
    }

    public void processPendingBlock()
    {
        // get first block
        BlockCoords block = pending.pollFirst();

        // and move it to network list
        network.add(block);

        // add chunk
        chunks.add(new ChunkCoords(block.x, block.z));

        // collect statistics
        tilesFound++;
        chunksFound = chunks.size();
        syncField("tilesFound");
        syncField("chunksFound");

        // check and schedule surrounding blocks
        checkAdjacentBlocks(block.x, block.y, block.z);
    }

    public void checkAdjacentBlocks(int x, int y, int z)
    {
        checkAndScheduleBlock(x - 1, y, z);
        checkAndScheduleBlock(x + 1, y, z);
        checkAndScheduleBlock(x, y - 1, z);
        checkAndScheduleBlock(x, y + 1, z);
        checkAndScheduleBlock(x, y, z - 1);
        checkAndScheduleBlock(x, y, z + 1);
    }

    public void checkAndScheduleBlock(int x, int y, int z)
    {
        if (worldObj.getBlockTileEntity(x, y, z) instanceof IEnergyTile)
        {
            BlockCoords b = new BlockCoords(x, y, z);
            if (!network.contains(b)) pending.add(b);
        }
    }

    public void cleanupGeometry()
    {
        network.clear();
        pending.clear();
        chunks.clear();
    }

    public void releaseChunks()
    {
        for (Ticket t : tickets) ForgeChunkManager.releaseTicket(t);
        tickets.clear();
    }

    public void reset()
    {
        if (worldObj.isRemote == false)
        {
            cleanupGeometry();
            releaseChunks();
        }
        tilesFound = 0; // reset statistics
        chunksForced = 0;
        chunksFound = 0;
        ticketsUsed = 0;
        state = 0; // start over
    }

    /**
     * Reads a tile entity from NBT.
     */
    @Override
    public void readFromNBT(NBTTagCompound nbt)
    {
        super.readFromNBT(nbt);

        // Read inventory stacks from NBT.

        NBTTagList items = nbt.getTagList("Items");
        inventory = new ItemStack[getSizeInventory()];

        for (int i = 0; i < items.tagCount(); i++)
        {
            NBTTagCompound item = (NBTTagCompound) items.tagAt(i);
            byte slot = item.getByte("Slot");

            if (slot >= 0 && slot < inventory.length)
            {
                inventory[slot] = ItemStack.loadItemStackFromNBT(item);
            }
        }

        // Read rest parameters

        // migration from old "mode"
        if (nbt.hasKey("mode"))
        {
            int mode = nbt.getInteger("mode");
            scan = mode >= 1; // network scan is enabled in modes 1 and 2
            if (mode == 1) area = 0; // area should be 1x1 in network mode
            nbt.removeTag("mode");
        }
        else
        {
            scan = nbt.getBoolean("scan");
            area = nbt.getInteger("area");
        }

        energy = nbt.getInteger("energy");

        // Calculate parameters depending on installed upgrades

        applyUpgrades();
    }

    /**
     * Writes a tile entity to NBT.
     */
    @Override
    public void writeToNBT(NBTTagCompound nbt)
    {
        super.writeToNBT(nbt);

        // Write inventory stacks to NBT.

        NBTTagList items = new NBTTagList();

        for (int i = 0; i < inventory.length; i++)
        {
            if (inventory[i] != null)
            {
                NBTTagCompound item = new NBTTagCompound();
                item.setByte("Slot", (byte) i);
                inventory[i].writeToNBT(item);
                items.appendTag(item);
            }
        }

        nbt.setTag("Items", items);

        // Write rest parameters

        nbt.setBoolean("scan", scan);
        nbt.setInteger("area", area);
        nbt.setInteger("energy", energy);
    }

    // IEnergySink implementation

    public boolean isReadyForEnergy()
    {
        return energyEnabled == true && isInvalid() == false && isAddedToEnergyNet() == true && initialized == true;
    }

    @Override
    public boolean acceptsEnergyFrom(TileEntity emitter, Direction direction)
    {
        return true;
    }

    @Override
    public boolean isAddedToEnergyNet()
    {
        return addedToEnergyNet;
    }

    @Override
    public int demandsEnergy()
    {
        return isReadyForEnergy() == true ? maxEnergy - energy : 0;
    }

    @Override
    public int injectEnergy(Direction directionFrom, int amount)
    {
        if (isReadyForEnergy() == false) return amount;

        // Check voltage
        if (amount > getMaxSafeInput())
        {
            // Explode
            invalidate();
            worldObj.setBlockToAir(xCoord, yCoord, zCoord);
            worldObj.createExplosion(null, xCoord + 0.5D, yCoord + 0.5D, zCoord + 0.5D, 1.25F, true);
            return 0; // and eat all energy
        }

        int n = Math.min(maxEnergy - energy, amount);
        setEnergy(energy + n);
        return amount - n;
    }

    @Override
    public int getMaxSafeInput()
    {
        return isReadyForEnergy() == true ? maxInput : Integer.MAX_VALUE;
    }

    // IInventory implementation

    /**
     * Returns the number of slots in the inventory.
     */
    @Override
    public int getSizeInventory()
    {
        return inventory.length;
    }

    /**
     * Returns the stack in slot i
     */
    @Override
    public ItemStack getStackInSlot(int i)
    {
        return inventory[i];
    }

    /**
     * Removes from an inventory slot (first arg) up to a specified number (second arg) of items and returns them in a new stack.
     */
    @Override
    public ItemStack decrStackSize(int i, int count)
    {
        if (inventory[i] == null) return null;

        ItemStack stack;

        if (inventory[i].stackSize <= count)
        {
            stack = inventory[i];
            inventory[i] = null;
            return stack;
        }

        stack = inventory[i].splitStack(count);
        if (inventory[i].stackSize == 0) inventory[i] = null;
        return stack;
    }

    /**
     * When some containers are closed they call this on each slot, then drop whatever it returns as an EntityItem - like when you close a workbench GUI.
     */
    @Override
    public ItemStack getStackInSlotOnClosing(int i)
    {
        ItemStack stack = inventory[i];
        inventory[i] = null;
        return stack;
    }

    /**
     * Sets the given item stack to the specified slot in the inventory (can be crafting or armor sections).
     */
    @Override
    public void setInventorySlotContents(int i, ItemStack stack)
    {
        inventory[i] = stack;

        if (stack != null && stack.stackSize > getInventoryStackLimit())
        {
            stack.stackSize = getInventoryStackLimit();
        }
    }

    /**
     * Returns the name of the inventory.
     */
    @Override
    public String getInvName()
    {
        return "container.networkAnchor";
    }

    @Override
    public boolean isInvNameLocalized()
    {
        return false;
    }

    /**
     * Returns the maximum stack size for a inventory slot. Seems to always be 64, possibly will be extended. *Isn't this more of a set than a get?*
     */
    @Override
    public int getInventoryStackLimit()
    {
        return 64;
    }

    /**
     * Do not make give this method the name canInteractWith because it clashes with Container
     */
    @Override
    public boolean isUseableByPlayer(EntityPlayer player)
    {
        return worldObj.getBlockTileEntity(xCoord, yCoord, zCoord) != this ? false : player.getDistanceSq(xCoord + 0.5D, yCoord + 0.5D, zCoord + 0.5D) <= 64.0D;
    }

    @Override
    public void openChest()
    {
        // do nothing
    }

    @Override
    public void closeChest()
    {
        // do nothing
    }

    /**
     * Returns true if automation is allowed to insert the given stack (ignoring stack size) into the given slot.
     */
    @Override
    public boolean isStackValidForSlot(int i, ItemStack stack)
    {
        return stack != null && (UpgradeDictionary.get(stack) != null || getBattery(stack) != null);
    }

    public static IElectricItem getBattery(ItemStack stack)
    {
        if (stack == null) return null;
        Item item = stack.getItem();
        if (item == null || item instanceof IElectricItem == false) return null;
        IElectricItem electricItem = (IElectricItem) item;
        if (electricItem.canProvideEnergy(stack) == false) return null;
        return electricItem;
    }

    /**
     * Called when an the contents of an Inventory change, usually
     */
    @Override
    public void onInventoryChanged()
    {
        super.onInventoryChanged();
        applyUpgrades();
    }

    /**
     * Update parameters depending on installed upgrades
     */
    public void applyUpgrades()
    {
        // Counters with initial parameters for the device
        int cTier = 1;
        int cSpeed = 1;
        int cStorage = 10000;

        // Collecting tier upgrades first (e.g. transformers)
        for (ItemStack stack : inventory)
        {
            if (stack == null) continue;
            UpgradeModule module = UpgradeDictionary.get(stack);
            if (module == null) continue;
            if (module.type == UpgradeDictionary.TIER)
                cTier += module.amplifier * stack.stackSize;
        }

        // Iterate through upgrade slots and collect other upgrades
        for (ItemStack stack : inventory)
        {
            if (stack == null) continue;
            UpgradeModule module = UpgradeDictionary.get(stack);
            if (module == null) continue;
            if (cTier < module.tier) continue; // skip module which have not enough tier
            if (module.type == UpgradeDictionary.SPEED)
                cSpeed += module.amplifier * stack.stackSize;
            else if (module.type == UpgradeDictionary.STORAGE)
                cStorage += module.amplifier * stack.stackSize;
        }

        // Actually apply parameters
        tier = cTier;
        blocksPerTick = (int) Math.pow(2, cSpeed);
        maxInput = (int) Math.pow(2, cTier * 2 + 3);
        setMaxEnergy(cStorage);

        // Cut overcharge (usually when removing storage upgrades)
        if (energy > maxEnergy) setEnergy(maxEnergy);
    }

    @Override
    public void onNetworkEvent(EntityPlayer player, int event)
    {
        switch (event) // event = button.id
        {
            case 0: // Network scan
                scan = !scan;
                break;
            case 1: // A-
                if (area > 0) area--;
                break;
            case 2: // A+
                area++;
                break;
            case 3: // Restart
                reset();
                break;
        }
    }

    @Override
    public void onNetworkUpdate(String field)
    {
        // no special actions
    }

    @Override
    public List<String> getNetworkedFields()
    {
        return networkedFields;
    }
}
