package mods.NetworkAnchor;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;
import java.util.logging.Logger;

import net.minecraft.block.Block;
import net.minecraft.client.Minecraft;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.util.StringTranslate;

import com.google.common.base.Charsets;

import cpw.mods.fml.common.Loader;
import cpw.mods.fml.common.registry.LanguageRegistry;

public class LangHelper
{
    public static final String MASTER_LANG = "en_US";
    private final String modid; // For copy&paste purpose
    private final Logger log;
    private final Set<?> languages;
    private Set<String> master;

    public LangHelper()
    {
        modid = Loader.instance().activeModContainer().getModId();
        log = Logger.getLogger(modid);
        languages = StringTranslate.getInstance().getLanguageList().keySet();
        master = null;
    }

    public static Properties reduce(Set<String> master, Properties slave)
    {
        Properties result = new Properties();
        Set<String> keys = slave.stringPropertyNames();
        for (String key : keys)
        {
            if (master.contains(key))
            {
                result.setProperty(key, slave.getProperty(key));
            }
        }
        return result;
    }

    public static Properties loadFromStream(InputStream input)
    {
        if (input == null) return null;
        Properties prop = new Properties();
        BufferedReader buf = new BufferedReader(new InputStreamReader(input, Charsets.UTF_8));
        try
        {
            // Read first line
            String line = buf.readLine();
            if (line != null)
            {
                // Need own parser to get rid of UTF-8 BOM in the first line
                if (line.startsWith("\uFEFF")) line = line.substring(1);

                // Process lines
                do
                {
                    // Skip if comment
                    if (!line.startsWith("#"))
                    {
                        // Find separator
                        int n = line.indexOf('=');

                        // If line invalid - skip it
                        if (n != -1)
                        {
                            // Extract key and value, and put them into translation table
                            String key = line.substring(0, n);
                            String value = line.substring(n + 1);
                            prop.setProperty(key, value);
                        }
                    }

                    // Next line
                    line = buf.readLine();
                }
                while (line != null);
            } // if line != null
        }
        catch (IOException e1)
        {
            e1.printStackTrace();
            prop = null;
        }
        finally
        {
            try
            {
                buf.close();
            }
            catch (IOException e2)
            {
                e2.printStackTrace();
            }
        }
        return prop;
    }

    public static Properties loadFromResource(String name)
    {
        return loadFromStream(LangHelper.class.getResourceAsStream(name));
    }

    public static Properties loadFromFile(String filename)
    {
        try
        {
            return loadFromStream(new FileInputStream(filename));
        }
        catch (FileNotFoundException e)
        {
            e.printStackTrace();
            return null;
        }
    }

    public Set<String> getMaster()
    {
        return master;
    }

    public void setMaster(Set<String> master)
    {
        this.master = master;
    }

    public void setMasterName(Object obj, String value)
    {
        // Cowardly stolen from LanguageRegistry.
        String key;
        if (obj instanceof Item)
            key = ((Item) obj).getUnlocalizedName();
        else if (obj instanceof Block)
            key = ((Block) obj).getUnlocalizedName();
        else if (obj instanceof ItemStack)
            key = ((ItemStack) obj).getItem().getUnlocalizedName((ItemStack) obj);
        else
            throw new IllegalArgumentException(String.format("Illegal object for naming %s", obj));
        setMasterName(key + ".name", value);
    }

    public void setMasterString(String key, String value)
    {
        if (master == null) master = new HashSet<String>();
        master.add(key);
        LanguageRegistry.instance().addStringLocalization(key, value);
    }

    public void loadResources()
    {
        loadResources("/mods/" + modid + "/lang/");
    }

    public void loadResources(String subdir)
    {
        String path = subdir;

        // Make sure subdir ends with separator
        if (path != "" && !path.endsWith("/")) path += "/";

        // Firstly try to load master localization file
        Properties p = loadFromResource(path + MASTER_LANG + ".txt");
        if (master == null || master.isEmpty())
        {
            if (p == null || p.isEmpty())
            {
                log.warning("Master localization not specified and could not be found into resources, aborting...");
                return;
            }
            master = p.stringPropertyNames();
        }
        else if (p != null && !p.isEmpty())
        {
            p = reduce(master, p);
        }

        // Add or override master localization
        if (p != null && !p.isEmpty())
            LanguageRegistry.instance().addStringLocalization(p, MASTER_LANG);

        // There is no way to enumerate resources, so using trial&error
        for (Object lang : languages)
        {
            // Skip master language as it already loaded
            if (lang == MASTER_LANG) continue;

            // Load and process resource
            p = loadFromResource(path + lang + ".txt");
            if (p == null || p.isEmpty()) continue;
            p = reduce(master, p);
            if (p == null || p.isEmpty()) continue;
            LanguageRegistry.instance().addStringLocalization(p, lang.toString());
        }
    }

    public void loadUserFiles()
    {
        loadUserFiles("mods/" + modid + "/lang/");
    }

    public void loadUserFiles(String subdir)
    {
        // Get Minecraft path
        String path = Minecraft.getMinecraftDir().getAbsolutePath();

        // Finish if it empty
        if (path == null || path == "")
        {
            log.warning("Cannot obtain minecraft path - translations will be unavailable.");
            return;
        }

        // Make sure it ends with separator
        if (!path.endsWith(File.separator)) path += File.separator;

        path += subdir;

        // Make sure it ends with separator, again
        if (!path.endsWith(File.separator)) path += File.separator;

        // Build translations path as following
        File langPath = new File(path);
        log.info(String.format("Trying to look for language files into: \"%s\"", langPath.getAbsolutePath())); // Emit info
        File[] files = langPath.listFiles(); // List files
        if (files == null || files.length == 0) // Finish if failed
        {
            log.info("... directory either not exists or empty, done with that.");
            return;
        }

        // Firstly try to load master localization file
        Properties p = loadFromFile(path + MASTER_LANG + ".txt");
        if (master == null || master.isEmpty())
        {
            if (p == null || p.isEmpty())
            {
                log.warning("Master localization not specified and could not be found into user files, aborting...");
                return;
            }
            master = p.stringPropertyNames();
        }
        else if (p != null && !p.isEmpty())
        {
            p = reduce(master, p);
        }

        // Add or override master localization
        if (p != null && !p.isEmpty())
            LanguageRegistry.instance().addStringLocalization(p, MASTER_LANG);

        // Iterate through files and try to load them
        for (File file : files)
        {
            // Skip invalid entries
            if (file == null) continue;

            // Use file name as language identifier
            String filename = file.getName();

            // Split filename into filename which is language key, and extension which determines whether file is xml
            int n = filename.indexOf('.');
            if (n == -1) continue;

            // Extract language id
            filename = filename.substring(0, n);

            // Check whether filename is valid language id
            if (!languages.contains(filename)) continue;

            // Load and process file
            p = loadFromFile(file.getAbsolutePath());
            if (p == null || p.isEmpty()) continue;
            p = reduce(master, p);
            if (p == null || p.isEmpty()) continue;
            LanguageRegistry.instance().addStringLocalization(p, filename);
        }
    }

    public void loadAll()
    {
        loadResources();
        loadUserFiles();
        if (master == null || master.isEmpty())
        {
            log.severe("Master localization is unspecified and it not found due resource and user file load.");
        }
    }
}
