package mods.ElectricRails;

import ic2.api.Direction;
import ic2.api.energy.event.EnergyTileLoadEvent;
import ic2.api.energy.event.EnergyTileUnloadEvent;
import ic2.api.energy.tile.IEnergySink;

import java.util.Random;

import net.minecraft.block.Block;
import net.minecraft.client.Minecraft;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraftforge.common.MinecraftForge;
import cpw.mods.fml.relauncher.Side;
import cpw.mods.fml.relauncher.SideOnly;

/**
 * Simple energy class. Does not support network energy synchronization see TileEntityNetworkedSink.
 */
public abstract class TileEntitySink extends TileEntityBase implements IEnergySink
{
    //public static final int maxEnergy = 20;
    public boolean addedToEnergyNet = false;
    public int energy = 0;
    public double consumed = 0.0D;
    public Random rand = new Random();

    /**
     * Returns carrier of this block, e.g. same block but without wires/circuitry.
     */
    public abstract ItemStack getCarrier();

    /**
     * Returns capacity of internal storage.
     */
    public int getCapacity()
    {
        return 20;
    }

    /**
     * Determines whether there is enough energy.
     *
     * @param amount of energy that expected to consume.
     * @return true if there is enough energy, or false otherwise.
     */
    public boolean canConsume(double amount)
    {
        return energy >= consumed + amount;
    }

    /**
     * Allows to consume energy and possibly stores fractional parts to bill.
     *
     * @param amount of energy to consume
     */
    public void consumeEnergy(double amount)
    {
        double total = consumed + amount; // Count bill
        int n = (int) Math.min(total, energy); // Try to consume at least part of bill
        energy -= n; // Consume energy
        consumed = total - n; // Write back remainder
    }

    /**
     * Shortcut function that forces to consume (flushes) calculated energy bill.
     */
    public void consumeEnergy()
    {
        consumeEnergy(0.0D);
    }

    /**
     * Spawns single spark into world
     *
     * @param startX of spark motion.
     * @param startY of spark motion.
     * @param startZ of spark motion.
     */
    @SideOnly(Side.CLIENT)
    public void doSpark(double startX, double startY, double startZ)
    {
        Minecraft.getMinecraft().effectRenderer.addEffect(new EntitySparksFX(worldObj, startX, startY, startZ, rand.nextGaussian() * 0.15D, rand.nextDouble() * 0.2D, rand.nextGaussian() * 0.15D));
    }

    /**
     * Spawns up to 20 sparks into world, depending on game settings.
     *
     * @param startX of spark motion.
     * @param startY of spark motion.
     * @param startZ of spark motion.
     */
    @SideOnly(Side.CLIENT)
    public void doSparks(double startX, double startY, double startZ)
    {
        // Count how much to spawn depending on game settings
        int i = (int) (Minecraft.getMinecraft().gameSettings.particleSetting / 2.10F * 20.0F);

        // Spawn particles in the world
        for (; i < 20; i++) doSpark(startX, startY, startZ);
    }

    /**
     * Spawn random sparks from block.
     *
     * @param height of block.
     */
    public void doRandomSparks()
    {
        // Count how much to spawn depending on game settings
        int i = (int) (Minecraft.getMinecraft().gameSettings.particleSetting / 2.10F * 20.0F);

        // Spawn particles in the world
        for (; i < 20; i++) doSpark(xCoord + rand.nextGaussian(), yCoord, zCoord + rand.nextGaussian());
    }

    @Override
    public void updateEntity()
    {
        // Make sure that we connected to energy net
        if (addedToEnergyNet == false)
        {
            MinecraftForge.EVENT_BUS.post(new EnergyTileLoadEvent(this));
            addedToEnergyNet = true;
        }
    }

    public void removeFromEnergyNet()
    {
        if (addedToEnergyNet == true)
        {
            MinecraftForge.EVENT_BUS.post(new EnergyTileUnloadEvent(this));
            addedToEnergyNet = false;
        }
    }

    @Override
    public void invalidate()
    {
        removeFromEnergyNet();
        super.invalidate();
    }

    @Override
    public void onChunkUnload()
    {
        removeFromEnergyNet();
        super.onChunkUnload();
    }

    @Override
    public void readFromNBT(NBTTagCompound nbt)
    {
        super.readFromNBT(nbt);
        energy = nbt.getInteger("energy");
        consumed = (float) nbt.getDouble("consumed");
        consumeEnergy();
    }

    @Override
    public void writeToNBT(NBTTagCompound nbt)
    {
        super.writeToNBT(nbt);
        consumeEnergy();
        nbt.setInteger("energy", energy);
        nbt.setDouble("consumed", consumed);
    }

    @Override
    public boolean acceptsEnergyFrom(TileEntity emitter, Direction direction)
    {
        return direction == Direction.YN;
    }

    @Override
    public boolean isAddedToEnergyNet()
    {
        return addedToEnergyNet;
    }

    @Override
    public int demandsEnergy()
    {
        return getCapacity() - energy;
    }

    @Override
    public int injectEnergy(Direction directionFrom, int amount)
    {
        // Check voltage
        if (amount > getMaxSafeInput())
        {
            onOvervoltage();
            return 0; // and eat all energy
        }

        int n = Math.min(getCapacity() - energy, amount);
        energy += n;
        return amount - n;
    }

    @Override
    public abstract int getMaxSafeInput();

    public void onOvervoltage()
    {
        invalidate();
        if (worldObj.isRemote)
        {
            // FX for client
            worldObj.playSoundEffect(xCoord + 0.5D, yCoord + 0.0625D, zCoord + 0.5D, "random.fizz", 0.5F, 2.6F + (worldObj.rand.nextFloat() - worldObj.rand.nextFloat()) * 0.8F);
            for (int i = 0; i < 8; i++)
            {
                worldObj.spawnParticle("largesmoke", xCoord + rand.nextGaussian(), yCoord, zCoord + rand.nextGaussian(), 0.0D, 0.0D, 0.0D);
            }
            doRandomSparks();
        }
        else
        {
            // Replace block with carrier, e.g. same block but without conductor(s)
            ItemStack carrier = getCarrier();
            if (carrier != null)
            {
                Block theBlock = Block.blocksList[carrier.itemID];
                if (theBlock != null && worldObj.setBlock(xCoord, yCoord, zCoord, carrier.itemID, carrier.getItemDamage(), 3) && worldObj.getBlockId(xCoord, yCoord, zCoord) == carrier.itemID)
                {
                    // TODO Not sure that it is safe to pass NULL as player
                    theBlock.onBlockPlacedBy(worldObj, xCoord, yCoord, zCoord, null, carrier);
                    theBlock.onPostBlockPlaced(worldObj, xCoord, yCoord, zCoord, carrier.getItemDamage());
                    return;
                }
            }

            // Set to air if fail
            worldObj.setBlockToAir(xCoord, yCoord, zCoord);
        }
    }
}
