package ic2backpackhud;

import java.lang.reflect.Field;

import org.lwjgl.opengl.GL11;

import ic2.api.item.IElectricItem;
import net.minecraft.client.Minecraft;
import net.minecraft.client.entity.EntityPlayerSP;
import net.minecraft.client.gui.ScaledResolution;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.WorldRenderer;
import net.minecraft.client.renderer.entity.RenderItem;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.ResourceLocation;
import net.minecraftforge.fml.common.eventhandler.SubscribeEvent;
import net.minecraftforge.fml.common.gameevent.TickEvent;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

@SideOnly(Side.CLIENT)
public class ClientTickHandler {
	private final Minecraft minecraft = Minecraft.getMinecraft();
	private final RenderItem itemRenderer = minecraft.getRenderItem();
	private int scaledWidth, scaledHeight;
	private long[] elapsedDrawTime = new long[4];
	private int[] currentCharge = new int[4], lastCharge = new int[4];
	private boolean[] isCharging = new boolean[4], isLowPower = new boolean[4];
	private final byte pixelNudge = 1;
	private final byte bgHeight = 21, bgWidth = 18;

	private static final ResourceLocation hudTextures = new ResourceLocation("IC2BackpackHUD:textures/sprites.png");

	private static final byte POS_HORIZ_LEFT = 0;
	private static final byte POS_HORIZ_RIGHT = 1;
	private static final byte POS_VERTICAL_TL = 2;
	private static final byte POS_VERTICAL_BL = 3;
	private static final byte POS_VERTICAL_BR = 4;
	private static final byte POS_VERTICAL_TR = 5;

	@SubscribeEvent
	public void tickRender(TickEvent.RenderTickEvent event) {
		EntityPlayerSP player = minecraft.thePlayer;

		if (minecraft.currentScreen != null || player == null) {
			return;
		}

		ScaledResolution scaledResolution = new ScaledResolution(minecraft);
		scaledWidth = scaledResolution.getScaledWidth();
		scaledHeight = scaledResolution.getScaledHeight();

		/**
		 * Armor Inventory Key
		 *
		 * Helmet     = armorInventory[3]
		 * Chestplate = armorInventory[2]
		 * Leggings   = armorInventory[1]
		 * Boots      = armorInventory[0]
		 */

		ItemStack[] armorInventory = player.inventory.armorInventory;
		boolean[] isValidArmor = new boolean[4];
		boolean hasValidArmor = false;
		byte armorPiecesWorn = 0;

		/* Detect worn armor pieces. */

		for (int slot = 0; slot < 4; slot++) {
			if (armorInventory[slot] != null) {
				if (IC2BackpackHUD.ic2Loaded && armorInventory[slot].getItem() instanceof IElectricItem || IC2BackpackHUD.showNonElectricItems && armorInventory[slot].getItem().isDamageable()
					&& armorInventory[slot].getItemDamage() >= 0) {
					armorPiecesWorn++;
					isValidArmor[slot] = true;
					hasValidArmor = true;
				}
			}
		}

		if (hasValidArmor) {
			int[] xCoord = new int[4];
			int[] yCoord = new int[4];

			for (byte count = 3, tempPiecesWorn = armorPiecesWorn; count >= 0; count--) {
				if (isValidArmor[count]) {
					switch (IC2BackpackHUD.hudPosition) {
						case POS_VERTICAL_TL:
							xCoord[count] = pixelNudge * 3;
							yCoord[count] = pixelNudge - 16 + (((armorPiecesWorn + 1) - tempPiecesWorn--) * bgHeight);
							break;
						case POS_VERTICAL_BL:
							xCoord[count] = pixelNudge * 3;
							yCoord[count] = scaledHeight - 16 - (pixelNudge * 3) - ((tempPiecesWorn-- - 1) * bgHeight);
							break;
						case POS_HORIZ_LEFT:
							xCoord[count] = (scaledWidth / 2 - 92) - tempPiecesWorn-- * bgWidth;
							yCoord[count] = scaledHeight - 16 - pixelNudge * 2;
							break;
						case POS_HORIZ_RIGHT:
							xCoord[count] = (scaledWidth / 2 + 94) + (armorPiecesWorn - tempPiecesWorn--) * bgWidth;
							yCoord[count] = scaledHeight - 16 - pixelNudge * 2;
							break;
						case POS_VERTICAL_BR:
							xCoord[count] = scaledWidth - pixelNudge - bgWidth;
							yCoord[count] = scaledHeight - 16 - (pixelNudge * 3) - ((tempPiecesWorn-- - 1) * bgHeight);
							break;
						case POS_VERTICAL_TR:
							xCoord[count] = scaledWidth - pixelNudge - bgWidth;
							yCoord[count] = pixelNudge - 16 + (((armorPiecesWorn + 1) - tempPiecesWorn--) * bgHeight);
							break;
						default:
							throw new IllegalStateException("Illegal Hud position: " + IC2BackpackHUD.hudPosition);
					}
				}
			}

			/* Draw armor pieces. */
			if (isValidArmor[3] && IC2BackpackHUD.showHelmetItems) {
				drawArmor(3, xCoord[3], yCoord[3]);
			}
			if (isValidArmor[2] && IC2BackpackHUD.showChestplateItems) {
				drawArmor(2, xCoord[2], yCoord[2]);
			}
			if (isValidArmor[1] && IC2BackpackHUD.showLeggingItems) {
				drawArmor(1, xCoord[1], yCoord[1]);
			}
			if (isValidArmor[0] && IC2BackpackHUD.showBootItems) {
				drawArmor(0, xCoord[0], yCoord[0]);
			}
		}
	}

	/**
	 * Return ItemStack damage as normalized integer between 0 and 14
	 */
	private int getChargeLevel(ItemStack armor) {
		if (armor.isItemDamaged()) {
			/*
			 * Vanilla item damage level is computed using a 13-value range.
			 * This introduces rounding errors for low max damage items when
			 * scaling to a 15-value range. This returns the correct zero-value
			 * item charge according to default item damage level checks.
			 */
			if ((int) Math.round(13.0D - armor.getItemDamage() * 13.0D / armor.getMaxDamage()) == 0) {
				return 0;
			} else {
				return (int) Math.round(15.0D - armor.getItemDamage() * 15.0D / armor.getMaxDamage());
			}
		} else {
			return 14;
		}
	}

	/**
	 * Draws a textured rectangle at the stored z-value. Args: x, y, u, v
	 */
	private void renderIconUsingOffset(int x, int y, int z, int uOffset, int vOffset, int u, int v) {
		WorldRenderer wr = Tessellator.getInstance().getWorldRenderer();
		wr.begin(GL11.GL_QUADS, DefaultVertexFormats.POSITION_TEX);
		float factor = 0.00390625F;
		wr.pos(x + 0, y + v, z).tex((uOffset + 0) * factor, (vOffset + v) * factor).endVertex();
		wr.pos(x + u, y + v, z).tex((uOffset + u) * factor, (vOffset + v) * factor).endVertex();
		wr.pos(x + u, y + 0, z).tex((uOffset + u) * factor, (vOffset + 0) * factor).endVertex();
		wr.pos(x + 0, y + 0, z).tex((uOffset + 0) * factor, (vOffset + 0) * factor).endVertex();
		Tessellator.getInstance().draw();
	}

	/**
	 * Draw armor item starting at given coordinate
	 */
	private void drawArmor(int armorNum, int xCoord, int yCoord) {
		GL11.glEnable(GL11.GL_BLEND);
		GL11.glBlendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);

		ItemStack itemStack = minecraft.thePlayer.inventory.armorInventory[armorNum];
		int chargeLevel = getChargeLevel(itemStack);
		float dura_red = 1.0F - chargeLevel / 14.0F;
		float dura_grn = chargeLevel / 14.0F;

		/* Draw status background */
		if (IC2BackpackHUD.showBackground) {
			minecraft.renderEngine.bindTexture(hudTextures);
			GL11.glColor4f(isCharging[armorNum] ? 1.0F : dura_red, isCharging[armorNum] ? 1.0F : dura_grn, 0.0F, 1.0F);
			renderIconUsingOffset(xCoord - 1, yCoord - 4, -100, 16, 0, bgWidth, bgHeight);
			GL11.glColor4f(1.0F, 1.0F, 1.0F, 1.0F);
		}

		/* Draw item */
		renderItemAndEffectIntoGUI(itemStack, xCoord, yCoord - 2);

		/* Draw status icons */
		minecraft.renderEngine.bindTexture(hudTextures);

		int elapsedTime = (int) (System.nanoTime() - elapsedDrawTime[armorNum]);
		if (elapsedTime > 500000000) {

			isCharging[armorNum] = false;
			isLowPower[armorNum] = false;

			if (IC2BackpackHUD.ic2Loaded && IC2BackpackHUD.showChargingFlasher) {
				if (itemStack.getItem() instanceof IElectricItem) {
					try {
						if (stackTagCompound == null) {
							hackTagCompound();
						}
						NBTTagCompound tag = (NBTTagCompound) stackTagCompound.get(itemStack);
						if (tag != null && tag.hasKey("charge")) {
							currentCharge[armorNum] = tag.getInteger("charge");
						}
					} catch (Exception e) {

					}

					if (currentCharge[armorNum] > lastCharge[armorNum]) {
						isCharging[armorNum] = true;
					}

					lastCharge[armorNum] = currentCharge[armorNum];
				}
			}

			if (IC2BackpackHUD.showLowPowerFlasher) {
				if (chargeLevel <= 2) {
					isLowPower[armorNum] = true;
				}
			}

			elapsedDrawTime[armorNum] = System.nanoTime();
		}

		if (isLowPower[armorNum] || isCharging[armorNum]) {
			GL11.glColor4f(1.0F, 1.0F, 1.0F, 1.0F - (chargeLevel == 0 ? 0 : (Math.abs(elapsedTime - 250000000) / 250000000.0F)));
			renderIconUsingOffset(xCoord, yCoord - 3, 0, 0, (isCharging[armorNum] ? 16 : 0), 16, 16);
			GL11.glColor4f(1.0F, 1.0F, 1.0F, 1.0F);
		}

		/* Draw durability or charge bar */
		GL11.glDisable(GL11.GL_TEXTURE_2D);
		GL11.glBegin(GL11.GL_QUADS);

		// Charge background
		GL11.glColor4f(0.03515625F, 0.0625F, 0.046875F, 1.0F); //Numbers
		GL11.glVertex3d(xCoord + 1, yCoord + 13, 0);
		GL11.glVertex3d(xCoord + 1, yCoord + 13 + 2, 0);
		GL11.glVertex3d(xCoord + 15, yCoord + 13 + 2, 0);
		GL11.glVertex3d(xCoord + 15, yCoord + 13, 0);

		// Depleted portion
		GL11.glColor4f(0.15625F, 0.24609375F, 0, 1.0F); //Magic numbers
		GL11.glVertex3d(xCoord + 1, yCoord + 13, 0);
		GL11.glVertex3d(xCoord + 1, yCoord + 13 + 1, 0);
		GL11.glVertex3d(xCoord + 14, yCoord + 13 + 1, 0);
		GL11.glVertex3d(xCoord + 14, yCoord + 13, 0);

		// Charged portion
		GL11.glColor4f(isCharging[armorNum] ? 1.0F : dura_red, isCharging[armorNum] ? 1.0F : dura_grn, 0.0F, 1.0F);
		GL11.glVertex3d(xCoord + 1, yCoord + 13, 0);
		GL11.glVertex3d(xCoord + 1, yCoord + 13 + 1, 0);
		GL11.glVertex3d(xCoord + 1 + chargeLevel, yCoord + 13 + 1, 0);
		GL11.glVertex3d(xCoord + 1 + chargeLevel, yCoord + 13, 0);
		GL11.glColor4f(1.0F, 1.0F, 1.0F, 1.0F);

		GL11.glEnd();
		GL11.glEnable(GL11.GL_TEXTURE_2D);

		GlStateManager.disableBlend();
	}

	private Field stackTagCompound = null;
	private void hackTagCompound() {
		try {
			stackTagCompound = ItemStack.class.getDeclaredField("stackTagCompound");
			stackTagCompound.setAccessible(true);
		} catch (Exception e) {
			IC2BackpackHUD.logger.fatal("Well that went badly :S", e);
			throw new RuntimeException("Error reflecting field", e);
		}
	}
	
	/**
	 * Renders the item's icon or block into the UI at the specified position.
	 * 
	 * Don't forget to do GlStateManager.disableBlend(); afterwards ;)
	 */
	private void renderItemAndEffectIntoGUI(ItemStack stack, int x, int y) {
        GlStateManager.enableDepth();
		itemRenderer.renderItemAndEffectIntoGUI(stack, x, y);
		GlStateManager.disableDepth();
	}
}