package ic2backpackhud;

import static ic2backpackhud.IC2BackpackHUD.MODID;

import java.lang.reflect.Field;
import java.util.Collections;

import org.apache.commons.lang3.reflect.FieldUtils;
import org.lwjgl.opengl.GL11;

import ic2.api.item.ElectricItem;
import ic2.api.item.IElectricItem;
import net.minecraft.client.Minecraft;
import net.minecraft.client.entity.EntityPlayerSP;
import net.minecraft.client.gui.GuiButton;
import net.minecraft.client.gui.GuiScreen;
import net.minecraft.client.gui.ScaledResolution;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.RenderItem;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.VertexBuffer;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.item.ItemStack;
import net.minecraft.util.ResourceLocation;
import net.minecraftforge.client.event.GuiScreenEvent.ActionPerformedEvent;
import net.minecraftforge.client.event.GuiScreenEvent.DrawScreenEvent;
import net.minecraftforge.energy.CapabilityEnergy;
import net.minecraftforge.energy.IEnergyStorage;
import net.minecraftforge.fml.client.GuiModList;
import net.minecraftforge.fml.client.config.GuiUtils;
import net.minecraftforge.fml.common.ModContainer;
import net.minecraftforge.fml.common.eventhandler.SubscribeEvent;
import net.minecraftforge.fml.common.gameevent.TickEvent.RenderTickEvent;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

@SideOnly(Side.CLIENT)
public class ClientTickHandler {
	private static final ResourceLocation hudTextures = new ResourceLocation(MODID, "textures/sprites.png");

	private static final byte pixelNudge = 1, bgHeight = 21, bgWidth = 18;
	private static final byte POS_HORIZ_LEFT = 0;
	private static final byte POS_HORIZ_RIGHT = 1;
	private static final byte POS_VERTICAL_TL = 2;
	private static final byte POS_VERTICAL_BL = 3;
	private static final byte POS_VERTICAL_BR = 4;
	private static final byte POS_VERTICAL_TR = 5;

	private final Minecraft minecraft = Minecraft.getMinecraft();
	private final RenderItem itemRenderer = minecraft.getRenderItem();

	private long[] elapsedDrawTime = new long[4];
	private int[] currentCharge = new int[4], lastCharge = new int[4];
	private boolean[] isCharging = new boolean[4], isLowPower = new boolean[4];

	private static class Armour {
		/** The armour ItemStack */
		public final ItemStack stack;
		/** ItemStack damage as normalised integer between 0 and 14 */
		public final int damage, charge;

		private static int mapCharge(double charge, double max) {
			return (int) (charge / max * 15);
		}

		public static Armour as_IC2(ItemStack stack) {
			assert IC2BackpackHUD.ic2Loaded && stack.getItem() instanceof IElectricItem;

			int damage;
			double charge;
			if (stack.hasTagCompound()) {
				damage = mapCharge(charge = ElectricItem.manager.getCharge(stack), ElectricItem.manager.getMaxCharge(stack));
			} else {
				charge = damage = 0;
			}

			return new Armour(stack, damage, (int) charge);
		}

		public static Armour as_Forge(ItemStack stack) {
			assert stack.hasCapability(CapabilityEnergy.ENERGY, null);

			IEnergyStorage energy = stack.getCapability(CapabilityEnergy.ENERGY, null);
			if (energy == null) throw new IllegalArgumentException("Invalid stack: "+stack);

			int charge = energy.getEnergyStored(); 
			return new Armour(stack, mapCharge(charge, energy.getMaxEnergyStored()), charge);
		}

		public static Armour as_Normal(ItemStack stack) {
			int damage;
			if (stack.isItemDamaged()) {
				/*
				 * Vanilla item damage level is computed using a 13-value range.
				 * This introduces rounding errors for low max damage items when
				 * scaling to a 15-value range. This returns the correct zero-value
				 * item charge according to default item damage level checks.
				 */
				if ((int) Math.round(13D - stack.getItemDamage() * 13D / stack.getMaxDamage()) == 0) {
					damage = 0;
				} else {
					damage = (int) Math.round(15D - stack.getItemDamage() * 15D / stack.getMaxDamage());
				}
			} else {
				damage = 14;
			}

			return new Armour(stack, damage, -1);
		}

		private Armour(ItemStack stack, int damage, int charge) {
			this.stack = stack;
			this.damage = damage;
			this.charge = charge;
		}

		public boolean hasCharge() {
			return charge >= 0;
		}
	}


	@SubscribeEvent
	public void tickRender(RenderTickEvent event) {
		EntityPlayerSP player = minecraft.thePlayer;

		if (!IC2BackpackHUD.isEnabled || minecraft.currentScreen != null || player == null) {
			return;
		}

		/**
		 * Armor Inventory Key
		 *
		 * Helmet     = armorInventory[3]
		 * Chestplate = armorInventory[2]
		 * Leggings   = armorInventory[1]
		 * Boots      = armorInventory[0]
		 */

		ItemStack[] armorInventory = player.inventory.armorInventory;
		Armour[] armours = new Armour[4];
		byte armorPiecesWorn = 0;

		/* Detect worn armor pieces. */

		for (int slot = 0; slot < 4; slot++) {
			if (armorInventory[slot] != null) {
				ItemStack stack = armorInventory[slot];
				Armour armour = null;
				
				if (IC2BackpackHUD.ic2Loaded && stack.getItem() instanceof IElectricItem) {
					armour = Armour.as_IC2(stack);
				} else if (stack.hasCapability(CapabilityEnergy.ENERGY, null)) {
					armour = Armour.as_Forge(stack);
				} else if (IC2BackpackHUD.showNonElectricItems && stack.isItemDamaged()) {
					armour = Armour.as_Normal(stack);
				}

				if (armour != null) {
					armorPiecesWorn++;
					armours[slot] = armour;
				}
			}
		}

		if (armorPiecesWorn > 0) {
			ScaledResolution scaledResolution = new ScaledResolution(minecraft);
			int scaledWidth = scaledResolution.getScaledWidth();
			int scaledHeight = scaledResolution.getScaledHeight();

			boolean[] itemConfig = new boolean[] {IC2BackpackHUD.showBootItems, IC2BackpackHUD.showLeggingItems,
													IC2BackpackHUD.showChestplateItems, IC2BackpackHUD.showHelmetItems};
			for (byte count = 3, tempPiecesWorn = armorPiecesWorn; count >= 0; count--) {
				if (itemConfig[count] && armours[count] != null) {
					int xCoord;
					int yCoord;

					switch (IC2BackpackHUD.hudPosition) {
						case POS_VERTICAL_TL:
							xCoord = pixelNudge * 3;
							yCoord = pixelNudge - 16 + (((armorPiecesWorn + 1) - tempPiecesWorn--) * bgHeight);
							break;
						case POS_VERTICAL_BL:
							xCoord = pixelNudge * 3;
							yCoord = scaledHeight - 16 - (pixelNudge * 3) - ((tempPiecesWorn-- - 1) * bgHeight);
							break;
						case POS_HORIZ_LEFT:
							int offset = player.inventory.offHandInventory[0] == null ? 92 : 122;
							xCoord = (scaledWidth / 2 - offset) - tempPiecesWorn-- * bgWidth;
							yCoord = scaledHeight - 16 - pixelNudge * 2;
							break;
						case POS_HORIZ_RIGHT:
							xCoord = (scaledWidth / 2 + 94) + (armorPiecesWorn - tempPiecesWorn--) * bgWidth;
							yCoord = scaledHeight - 16 - pixelNudge * 2;
							break;
						case POS_VERTICAL_BR:
							xCoord = scaledWidth - pixelNudge - bgWidth;
							yCoord = scaledHeight - 16 - (pixelNudge * 3) - ((tempPiecesWorn-- - 1) * bgHeight);
							break;
						case POS_VERTICAL_TR:
							xCoord = scaledWidth - pixelNudge - bgWidth;
							yCoord = pixelNudge - 16 + (((armorPiecesWorn + 1) - tempPiecesWorn--) * bgHeight);
							break;
						default:
							throw new IllegalStateException("Illegal Hud position: " + IC2BackpackHUD.hudPosition);
					}

					/* Draw armor pieces. */
					drawArmor(count, armours[count], xCoord, yCoord);
				}
			}
		}
	}

	/** Draws a textured rectangle at the stored z-value. Args: x, y, u, v */
	private static void renderIconUsingOffset(int x, int y, int z, int uOffset, int vOffset, int u, int v) {
		VertexBuffer wr = Tessellator.getInstance().getBuffer();
		wr.begin(GL11.GL_QUADS, DefaultVertexFormats.POSITION_TEX);
		final float factor = 0.00390625F;
		wr.pos(x + 0, y + v, z).tex((uOffset + 0) * factor, (vOffset + v) * factor).endVertex();
		wr.pos(x + u, y + v, z).tex((uOffset + u) * factor, (vOffset + v) * factor).endVertex();
		wr.pos(x + u, y + 0, z).tex((uOffset + u) * factor, (vOffset + 0) * factor).endVertex();
		wr.pos(x + 0, y + 0, z).tex((uOffset + 0) * factor, (vOffset + 0) * factor).endVertex();
		Tessellator.getInstance().draw();
	}

	/** Draw armor item starting at given coordinate */
	private void drawArmor(int armorNum, Armour armour, int xCoord, int yCoord) {
		GL11.glEnable(GL11.GL_BLEND);
		GL11.glBlendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);

		int chargeLevel = armour.damage;
		float dura_red = 1.0F - chargeLevel / 14.0F;
		float dura_grn = chargeLevel / 14.0F;

		/* Draw status background */
		if (IC2BackpackHUD.showBackground) {
			minecraft.renderEngine.bindTexture(hudTextures);
			GL11.glColor4f(isCharging[armorNum] ? 1.0F : dura_red, isCharging[armorNum] ? 1.0F : dura_grn, 0.0F, 1.0F);
			renderIconUsingOffset(xCoord - 1, yCoord - 4, -100, 16, 0, bgWidth, bgHeight);
			GL11.glColor4f(1.0F, 1.0F, 1.0F, 1.0F);
		}

		/* Draw item */
		renderItemAndEffectIntoGUI(armour.stack, xCoord, yCoord - 2);

		/* Draw status icons */
		minecraft.renderEngine.bindTexture(hudTextures);

		int elapsedTime = (int) (System.nanoTime() - elapsedDrawTime[armorNum]);
		if (elapsedTime > 500000000) {
			isCharging[armorNum] = false;
			isLowPower[armorNum] = false;

			if (IC2BackpackHUD.showChargingFlasher) {
				int charge;
				if (armour.hasCharge()) {			
					charge = currentCharge[armorNum] = armour.charge;
	
					if (currentCharge[armorNum] > lastCharge[armorNum]) {
						isCharging[armorNum] = true;
					}
				} else {
					charge = -1;
				}

				lastCharge[armorNum] = charge;
			}

			if (IC2BackpackHUD.showLowPowerFlasher) {
				if (chargeLevel <= 2) {
					isLowPower[armorNum] = true;
				}
			}

			elapsedDrawTime[armorNum] = System.nanoTime();
		}

		if (isLowPower[armorNum] || isCharging[armorNum]) {
			GL11.glColor4f(1.0F, 1.0F, 1.0F, 1.0F - (chargeLevel == 0 ? 0 : (Math.abs(elapsedTime - 250000000) / 250000000.0F)));
			renderIconUsingOffset(xCoord, yCoord - 3, 0, 0, (isCharging[armorNum] ? 16 : 0), 16, 16);
			GL11.glColor4f(1.0F, 1.0F, 1.0F, 1.0F);
		}

		/* Draw durability or charge bar */
		GL11.glDisable(GL11.GL_TEXTURE_2D);
		GL11.glBegin(GL11.GL_QUADS);

		// Charge background
		GL11.glColor4f(0.03515625F, 0.0625F, 0.046875F, 1.0F); //Numbers
		GL11.glVertex3d(xCoord + 1, yCoord + 13, 0);
		GL11.glVertex3d(xCoord + 1, yCoord + 13 + 2, 0);
		GL11.glVertex3d(xCoord + 15, yCoord + 13 + 2, 0);
		GL11.glVertex3d(xCoord + 15, yCoord + 13, 0);

		// Depleted portion
		GL11.glColor4f(0.15625F, 0.24609375F, 0, 1.0F); //Magic numbers
		GL11.glVertex3d(xCoord + 1, yCoord + 13, 0);
		GL11.glVertex3d(xCoord + 1, yCoord + 13 + 1, 0);
		GL11.glVertex3d(xCoord + 14, yCoord + 13 + 1, 0);
		GL11.glVertex3d(xCoord + 14, yCoord + 13, 0);

		// Charged portion
		GL11.glColor4f(isCharging[armorNum] ? 1.0F : dura_red, isCharging[armorNum] ? 1.0F : dura_grn, 0.0F, 1.0F);
		GL11.glVertex3d(xCoord + 1, yCoord + 13, 0);
		GL11.glVertex3d(xCoord + 1, yCoord + 13 + 1, 0);
		GL11.glVertex3d(xCoord + 1 + chargeLevel, yCoord + 13 + 1, 0);
		GL11.glVertex3d(xCoord + 1 + chargeLevel, yCoord + 13, 0);
		GL11.glColor4f(1.0F, 1.0F, 1.0F, 1.0F);

		GL11.glEnd();
		GL11.glEnable(GL11.GL_TEXTURE_2D);

		GlStateManager.disableBlend();
	}

	/**
	 * Renders the item's icon or block into the UI at the specified position.
	 * 
	 * Don't forget to do GlStateManager.disableBlend(); afterwards ;)
	 */
	private void renderItemAndEffectIntoGUI(ItemStack stack, int x, int y) {
        GlStateManager.enableDepth();
		itemRenderer.renderItemAndEffectIntoGUI(stack, x, y);
		GlStateManager.disableDepth();
	}


	private static final Field disableButton = FieldUtils.getDeclaredField(GuiModList.class, "disableModButton", true);
	private static final Field modContainer = FieldUtils.getDeclaredField(GuiModList.class, "selectedMod", true);

	@SubscribeEvent
	public void draw(DrawScreenEvent.Post event) {
		GuiScreen gui = event.getGui();

		if (gui instanceof GuiModList) {
			try {
				if (isRightMod(gui)) {
					GuiButton button = (GuiButton) FieldUtils.readField(disableButton, gui);

					if (button.isMouseOver()) {
						GuiUtils.drawHoveringText(Collections.singletonList((IC2BackpackHUD.isEnabled ? "Disable" : "Enable") + " IC2 Backpack HUD overlay"), 
													event.getMouseX(), event.getMouseY(), gui.width, gui.height, -1, minecraft.fontRendererObj);
					}
				}
			} catch (IllegalAccessException e) {
				IC2BackpackHUD.logger.error("Error drawing extra tooltip", e);
			}
		}
	}

	@SubscribeEvent
	public void buttonPressEvent(ActionPerformedEvent.Post event) {
		GuiScreen gui = event.getGui();

		if (gui instanceof GuiModList) {
			try {
				if (isRightMod(gui) && event.getButton() == FieldUtils.readField(disableButton, gui)) {
					IC2BackpackHUD.isEnabled = !IC2BackpackHUD.isEnabled;
				}
			} catch (IllegalAccessException e) {
				IC2BackpackHUD.logger.error("Error detecting disable button press", e);
			}
		}
	}

	private static boolean isRightMod(Object gui) throws IllegalAccessException {
		ModContainer mod = (ModContainer) FieldUtils.readField(modContainer, gui); 
		return mod != null && mod.getMod() == IC2BackpackHUD.instance;
	}
}