package fruitmods.fruitlibs;

// import java.io.File;
// import java.text.DecimalFormat;
import java.lang.Math;

import java.util.Arrays;
// import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
// import javax.imageio.ImageIO;


// let's try referencing them by name rather than by wildcard...
// import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.BufferBuilder;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.Tessellator;
import net.minecraft.client.renderer.entity.RenderManager;
import net.minecraft.client.renderer.vertex.DefaultVertexFormats;
import net.minecraft.client.renderer.texture.TextureManager;
// import net.minecraft.client.resources.IResourceManager;

// this appears to only be necessary for reloading the default font
// it'll have no effect on Unicode and so we can leave it out
// import net.minecraft.client.resources.ResourceManagerReloadListener;
// import net.minecraft.util.ChatAllowedCharacters;
import net.minecraft.util.ResourceLocation;

import net.minecraftforge.fml.client.FMLClientHandler;

// change required for MC r1.6.4
// RenderEngine no longer exists, FontRenderer appears to use this instead
// net.minecraft.client.gui.FontRenderer, for reference


import org.lwjgl.opengl.GL11;
// this is only used now for its constants stored as integers
// stuff like GL11.GL_TRIANGLE_STRIP or GL11.GL_SRC_ALPHA


// FruitDisplayLib contains all the useful functions for writing to the display
// box drawing, text rendering, etc.
// it also contains associated helpers, such as HSV to RGB conversion

public class FruitDisplayLib
{
  
  // changelog:
  
  // 1.1:
  // -added some resilience to Unicode characters - they now get displayed as question marks
  
  // 1.2a:
  // -altered the DrawShadedRect function to use direct GL calls instead of Tessellator.class calls
  // --this should fix the problem with only half of it being drawn
  // --along with providing a small speedup
  // -functions such as DrawShadedRect now take floats where they previously took doubles
  // --this was done because GL calls take floats and as such reduces the amount of typecasting I need to do
  
  // v1.2a compiles fine on MC r1.4.2, so no changes here.
  
  // v1.3 was a port to MC r1.4.6 (which moved all the classes around into separate packages)
  // -FruitDisplayLib is now in its own separate package (fruitmods.fruitlibs)
  
  // v1.4 was a port to MC r1.6.4, which mucked around with a LOT of MC internals
  // -source file stucture was radically re-written
  // -FruitDisplayLib now has a singular instance reachable via mod_FruitLibs
  // -this should fix all the static problems
  // -Unicode is now accepted - in fact, we only use Unicode now (the standard font is now unused)
  // -note that we don't use bidirectional fonts, because that's far too much hassle
  
  // v1.6 is a port to MC r1.12.2, which yet again scrambled FUCKING EVERYTHING
  
  
  private static final ResourceLocation[] unicodePageLocations = new ResourceLocation[256];
  
  public byte[] glyphWidth = new byte[65536];
  
  // un-necessary, we only use Unicode now
  // private final ResourceLocation locationFontTexture;
  
  private final TextureManager renderEngine;
  
  public FruitDisplayLib(TextureManager supplied_render_engine)
  {
    // yes, I know, hard-coded is evil
    // it's unlikely that this will change, however
    // this.locationFontTexture = new ResourceLocation("textures/font/ascii.png");
    
    this.renderEngine = supplied_render_engine;
    
    this.readGlyphSizes();
  }
  
  
  private void readGlyphSizes()
  {
    try
    {
      InputStream inputstream = FMLClientHandler.instance().getClient().getResourceManager().getResource(new ResourceLocation("font/glyph_sizes.bin")).getInputStream();
      inputstream.read(this.glyphWidth);
    }
    catch (IOException ioexception)
    {
      throw new RuntimeException(ioexception);
    }
  }

  
  public String getVersion()
  {
    return "1.6";
  }
  
  
  
  // yanked straight from FontRenderer.class
  private ResourceLocation getUnicodePageLocation(int page)
  {
    if (unicodePageLocations[page] == null)
    {
      unicodePageLocations[page] = new ResourceLocation(String.format("textures/font/unicode_page_%02x.png", new Object[] {Integer.valueOf(page)}));
    }
    
    return unicodePageLocations[page];
  }
  
  /**
   * Load one of the /font/glyph_XX.png into a new GL texture and store the texture ID in glyphTextureName array.
   */
  private void loadGlyphTexture(int page)
  {
    this.renderEngine.bindTexture(this.getUnicodePageLocation(page));
  }
  
  
  // this was originally private, but FontRenderer doing that seriously pissed me off
  // so renderScaledUnicodeChar is now a public function
  // enjoy
  
  public float renderScaledUnicodeChar(char unicode_char, float posX, float posY, float font_scaling, float[] RGBarray, float alpha)
  {
    if (glyphWidth[unicode_char] == 0)
    {
      return 0.0F;
    }
    else if(unicode_char == 32)
    {
      return 4.0F * font_scaling;
    }
    else
    {
      
      // if we want to change the colour of rendered text, this is where we do it
      // by creating a GlStateManager.color(R,G,B,A) here
      
      GlStateManager.color(RGBarray[0], RGBarray[1], RGBarray[2], alpha);
      
      int glyph_page = unicode_char / 256;
      
      this.loadGlyphTexture(glyph_page);
      
      // 4 times bitshifting to the right, ignoring sign
      // read: delete the lower 4 characters, shift the next 4 in their place
      // so, var4 = upper nibble
      int upper_width_nibble = this.glyphWidth[unicode_char] >>> 4;
      // logical AND with 15, or basically erasing all but the lower 4 bits
      // so, var5 = lower nibble
      int lower_width_nibble = this.glyphWidth[unicode_char] & 15;
      
      // below code filched from FontRenderer's getCharWidth function
      // thought it would allow me to remove the fudge factors, but no...
      /*
      if(lower_width_nibble > 7)
      {
	lower_width_nibble = 15;
	upper_width_nibble = 0;
      }
      */
      
      float f = (float)upper_width_nibble;
      float f1 = (float)(lower_width_nibble + 1.0F);
      // these bits above are going to cause real problems if I ever fully adapt it
      // to deal with higher-resolution fonts
      float f2 = (float)(unicode_char % 16 * 16) + f; // - 3.0F;
      // f2 = fractional co-ordinate of glyph on the x-axis
      // that - 2.0F on the end is a bastard empirical hack, since the initial code missed out
      // the first few pixels of the font
      float f3 = (float)((unicode_char & 255) / 16 * 16);
      // f3 = fractional co-ordinate of glyph on the y-axis
      float f4 = f1 - f - 0.02F;
      // float var10 = var7 - var6 + 4.0F;
      // likewise, this + 4.0F on the end is because it missed out the pixels at the end
      // var10 appears to be the width of the character
      GlStateManager.glBegin(GL11.GL_TRIANGLE_STRIP);
      // scaling for altered resolution texture packs
      // each glyph page is 256x256 pixels across and contains a 16x16 array of glyphs
      // so we need to fetch the size of said glyph page
      // easily done with the int getWidth() function in BufferedImage
      // we store it as a float, for ease of multiplication
      GlStateManager.glTexCoord2f(f2 / 256.0F, f3 / 256.0F);
      GlStateManager.glVertex3f(posX , posY, 0.0F);
      GlStateManager.glTexCoord2f(f2 / 256.0F, (f3 + 15.98F) / 256.0F);
      GlStateManager.glVertex3f(posX, posY + (7.99F * font_scaling), 0.0F);
      GlStateManager.glTexCoord2f((f2 + f4) / 256.0F, f3 / 256.0F);
      GlStateManager.glVertex3f(posX + (f4 / 2.0F) * font_scaling, posY, 0.0F);
      GlStateManager.glTexCoord2f((f2 + f4) / 256.0F, (f3 + 15.98F) / 256.0F);
      GlStateManager.glVertex3f(posX + (f4 / 2.0F) * font_scaling, posY + (7.99F * font_scaling), 0.0F);
      // though there are some hard-coded variables above, we should be alright
      // note that it's not completely image size independent, though
      GlStateManager.glEnd();
      return ((f1 - f) / 2.0F + 1.3F) * font_scaling;
      // was + 1.0F, is now +1.4F because unicode text is squished up by default
      // return ((var7 - var6) / 1.3F + 1.0F) * font_scaling;
      
      // return value is what we add on to the text position to space letters properly
    }
  }
  

  
  
  public void renderScaledStringAtPos(String input, float xstart, float ystart, float font_scaling, float[] RGBarray, float alpha, boolean hasShadow)
  {
    
    // shadow is broken for text sizes smaller than the default
    // you should draw a light grey rectangle behind it instead
    // the shadow flag is now completely ignored
    
    float writing_pos = xstart;
    float writing_offset = 0.0F;
    for(int i = 0; i < input.length(); i++)
    {
      writing_offset = renderScaledUnicodeChar(input.charAt(i), writing_pos, ystart, font_scaling, RGBarray, alpha);
      // writing_offset = renderScaledUnicodeChar(input.charAt(i), writing_pos, ystart, font_scaling, RGBarray, alpha);
      writing_pos += writing_offset;
      // was originally renderScaledUnicodeChar
      // but it completely refuses to get the correct texture co-ordinates
      // so we'll just use the default font instead...
    }
  }
  
  
  // originally from Gui.java
  // right, now it's getting the colour and transparency right
  // but not drawing the full rectangle
  // like this, it draws the upper-left, upper-right and lower-left vertices
  // so it missed out the last one
  public void drawShadedRect(float xpos, float ypos, float xsize, float ysize, float[] RGBarray, float alpha)
  {
    // it should be noted that this is for HUD elements only
    // they are given a Z-depth of zero
    
    // something broke in MC r1.12.2 whereby it just shows freaky triangles stretching way off the screen
    // let's just copy the nameplate background render code from RendererHax
    
    
    Tessellator tess = Tessellator.getInstance();
    BufferBuilder bufferBuilder = tess.getBuffer();
    
    GlStateManager.enableBlend();
    GlStateManager.disableTexture2D();
    GlStateManager.blendFunc(GL11.GL_SRC_ALPHA, GL11.GL_ONE_MINUS_SRC_ALPHA);
    // GlStateManager.color(RGBarray[0], RGBarray[1], RGBarray[2], alpha);
    
    bufferBuilder.begin(7, DefaultVertexFormats.POSITION_COLOR);
    bufferBuilder.pos(xpos, ypos, 0.0F).color(RGBarray[0], RGBarray[1], RGBarray[2], alpha).endVertex();
    bufferBuilder.pos(xpos, ypos + ysize, 0.0F).color(RGBarray[0], RGBarray[1], RGBarray[2], alpha).endVertex();
    bufferBuilder.pos(xpos + xsize, ypos + ysize, 0.0F).color(RGBarray[0], RGBarray[1], RGBarray[2], alpha).endVertex();
    bufferBuilder.pos(xpos + xsize, ypos, 0.0F).color(RGBarray[0], RGBarray[1], RGBarray[2], alpha).endVertex();
    tess.draw();

    
    
    GlStateManager.enableTexture2D();
    GlStateManager.disableBlend();
  }
  
  // we now need one to draw a hollow rectangle
  // simplest is to call DrawShadedRect 4 times
  
  public void drawHollowRect(float xpos, float ypos, float xsize, float ysize, float thickness, float[] RGBarray, float alpha)
  {
    // draw each "line" in turn, making sure not to overlap
    // otherwise the corners will end up with half the transparency and look weird
    // top line
    drawShadedRect(xpos, ypos, xsize - thickness, thickness, RGBarray, alpha);
    
    // right line
    drawShadedRect(xpos + xsize - thickness, ypos, thickness, ysize - thickness, RGBarray, alpha);
    
    // bottom line
    drawShadedRect(xpos + thickness, ypos + ysize - thickness, xsize - thickness, thickness, RGBarray, alpha);
    
    // left line
    drawShadedRect(xpos, ypos + thickness, thickness, ysize - thickness, RGBarray, alpha);
  }
  
  
  
  // we'll be giving the item condition as a hue
  // this function will make my life a lot easier
  
  public float[] HSVtoRGB(float hue, float saturation, float value)
  {
    // acceptable input values: hue = 0 to 360, saturation and value = 0 to 1
    
    // special case: if hue == -1.0F, return pure white
    
    // from the wiki page on HSV colour space:
    // [chroma] C = V * S
    // [hue_prime] H' = H / 60 (in degrees)
    // [alt_component] X = C(1 - abs((H' % 2) - 1))
    // [value_match] m = V - C
    // (R_1, G_1, B_1) = (C, X, 0) for 0 =< H' < 1
    // (X, C, 0) for 1 =< H' < 2
    // (0, C, X) for 2 =< H' < 3
    // (0, X, C) for 3 =< H' < 4
    // (X, 0, C) for 4 =< H' < 5
    // (C, 0, X) for 5 =< H' < 6
    // (0, 0, 0) otherwise, since it's undefined
    // (R, G, B) = (R_1 + m, G_1 + m, B_1 + m)
    
    // in fact, let's assign "otherwise" to (1, 1, 1)
    // so that we can take advantage of the "undefined" or NULL output from utility armours
    // RGB values outputted thus will range from 0 to 1 - perfect for my needs
    
    float[] result = {1.0F, 1.0F, 1.0F};
    // we leave the default as white to avoid putting in another if check
    // not technically correct, but it makes my life a little bit easier
    // this means that we can set <item>_fraction to 1.1F
    // and get white text - good for non-damageable items
    
    
    if(hue == -1.0F)
    {
      return result;
    }
    
    if(hue == 0.0F)
    {
      result[0] = 0.0F;
      result[1] = 0.0F;
      result[2] = 0.0F;
      return result;
    }
    // another break from technical correctness here
    // when hue is exactly zero (in the case of depleted armour)
    // we'll return black instead of the FF0000 we would normally have gotten
    
    float chroma = value * saturation;
    float hue_prime = hue / 60.0F;
    float alt_component = chroma * (1.0F - Math.abs((hue_prime % 2.0F) - 1.0F));
    
    float value_match = value - chroma;
    
    boolean isValidColour = true;
    
    if(0.0F <= hue_prime && hue_prime < 1.0F)
    {
      result[0] = chroma;
      result[1] = alt_component;
      result[2] = 0.0F;
    }
    else if(1.0F <= hue_prime && hue_prime < 2.0F)
    {
      result[0] = alt_component;
      result[1] = chroma;
      result[2] = 0.0F;
    }
    else if(2.0F <= hue_prime && hue_prime < 3.0F)
    {
      result[0] = 0.0F;
      result[1] = chroma;
      result[2] = alt_component;
    }
    else if(3.0F <= hue_prime && hue_prime < 4.0F)
    {
      result[0] = 0.0F;
      result[1] = alt_component;
      result[2] = chroma;
    }
    else if(4.0F <= hue_prime && hue_prime < 5.0F)
    {
      result[0] = alt_component;
      result[1] = 0.0F;
      result[2] = chroma;
    }
    else if(5.0F <= hue_prime && hue_prime < 6.0F)
    {
      result[0] = chroma;
      result[1] = 0.0F;
      result[2] = alt_component;
    }
    else
    {
      isValidColour = false;
    }
    
    if(isValidColour)
    {
      for(int i = 0; i < 3; i++)
      {
	result[i] = result[i] + value_match;
	// I really miss C++'s STL vector class
	// because I could use i < result.size() here for the ending condition
      }
    }
    
    return result;
  }
  
  
  // I think that OptiFine is overwriting getStringWidth and/or getCharWidth from FontRenderer
  // since Minecraft thinks that the actual string length changes if you use a texture pack
  // so I'll just copy out the unmodified ones and stick them here
  
  public float getStringWidth(String string_to_measure)
  {
    if (string_to_measure == null)
    {
      return 0;
    }
    else
    {
      float var2 = 0;
      boolean var3 = false;
      
      for (int var4 = 0; var4 < string_to_measure.length(); ++var4)
      {
	char var5 = string_to_measure.charAt(var4);
	float var6 = getCharWidth(var5);
	
	if (var6 < 0 && var4 < string_to_measure.length() - 1)
	{
	  ++var4;
	  var5 = string_to_measure.charAt(var4);
	  
	  if (var5 != 108 && var5 != 76)
	  {
	    if (var5 == 114 || var5 == 82)
	    {
	      var3 = false;
	    }
	  }
	  else
	  {
	    var3 = true;
	  }
	  
	  var6 = getCharWidth(var5);
	}
	
	var2 += var6;
	
	if (var3)
	{
	  var2 = var2 + 1.0F;
	}
      }
      
      return var2;
    }
  }
  
  /**
   * Returns the width of this character as rendered.
   */
  public float getCharWidth(char char_to_measure)
  {
    
    if (char_to_measure == 167)
    {
      return -1.0F;
    }
    else if (char_to_measure == 32)
    {
      return 4.0F;
    }
    else
    {
      // we never actually used this variable - not sure why it's here...
      // int var2 = ChatAllowedCharacters.allowedCharacters.indexOf(par1);
      
      if (this.glyphWidth[char_to_measure] != 0)
      {
	int j = this.glyphWidth[char_to_measure] >>> 4;
	int k = this.glyphWidth[char_to_measure] & 15;
	
	if (k > 7)
	{
	  k = 15;
	  j = 0;
	}
	
	++k;
	return (float)(k - j) / 2.0F + 1.3F;
      }
      else
      {
	return 0.0F;
      }
    }
  }
  
}
